package org.swift.util.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.swift.func.IProcessor;
import org.swift.util.IThreadPool;


/**
 * 功能描述：线程池
 */
public class ThreadPool implements IThreadPool {
	private ExecutorService threadPool = null;
	private List<Worker> workers;

	private class Worker implements Runnable {
		private IProcessor processor;
		private volatile boolean running;

		public Worker(IProcessor processor) {
			this.processor = processor;
			running = true;
			synchronized (workers) {
				workers.add(this);
			}
		}

		private synchronized boolean isRunning() {
			return running;
		}

		public void run() {
			if (processor.init()) {
				while (isRunning() && processor.cycle()) {
				}
			}
			processor.destroy();
			synchronized (workers) {
				workers.remove(this);
			}
			synchronized (this) {
				running = false;
				notify();
			}
		}

		public void close() {
			synchronized (this) {
				if (!running) {
					return;
				}
				running = false;
				try {
					wait();
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		}
	}

	public boolean addWorker(IProcessor processor) {
		if (processor == null || !isStart()) {
			return false;
		}
		threadPool.submit(new Worker(processor));
		return true;
	}

	public boolean addWorkers(IProcessor processor, int threadCount) {
		for (int count = 0; count < threadCount; count++) {
			if (!addWorker(processor)) {
				return false;
			}
		}
		return true;
	}

	public void start() {
		if (threadPool == null) {
			workers = new ArrayList<Worker>();
			threadPool = Executors.newCachedThreadPool();
		}
	}

	public void stop() {
		if (threadPool != null) {
			threadPool.shutdown();
			threadPool = null;
		}
		while (workers.size() > 0) {
			Worker worker = null;
			synchronized (workers) {
				worker = workers.get(0);
			}			
			if (worker != null) {
				worker.close();
			}
		}
	}

	public boolean isStart() {
		if (threadPool == null || threadPool.isShutdown()) {
			return false;
		}
		return true;
	}
}
