package sinosoftgz.utils.http;

import sinosoftgz.utils.http.entity.HttpRequest;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Liang Wenxu
 * @since 2019/5/14
 */
public interface HttpClientService {
    String URL_PLACEHOLDER_START = "{";
    String URL_PLACEHOLDER_END = "}";
    String ARRAY_IDX_START = "[";
    String ARRAY_IDX_END = "]";

    Pattern urlParamPattern = Pattern.compile(String.format("\\%s(.*?)\\%s", URL_PLACEHOLDER_START, URL_PLACEHOLDER_END));
    // 数组下标匹配正则
    Pattern arrayIdxPattern = Pattern.compile(String.format("^.*?\\%s(\\d+)\\%s$", ARRAY_IDX_START, ARRAY_IDX_END));


    /**
     * 发起get请求，由实现类实现
     * @param httpRequest 请求对象，其中的URL为已处理好占位符及url参数的url
     * @return
     */
    byte[] doGet(HttpRequest httpRequest) throws IOException;

    /**
     * 发起post请求，由实现类实现
     * @param httpRequest 请求对象，其中的URL为已处理好占位符及url参数的url
     * @return
     */
    byte[] doPost(HttpRequest httpRequest);

    /**
     * 发起delete请求，由实现类实现
     * @param httpRequest 请求对象，其中的URL为已处理好占位符及url参数的url
     * @return
     */
    byte[] doDelete(HttpRequest httpRequest);

    /**
     * 发起put请求，由实现类实现
     * @param httpRequest 请求对象，其中的URL为已处理好占位符及url参数的url
     * @return
     */
    byte[] doPut(HttpRequest httpRequest);

    /**
     * 发起get请求，实现类可选择性Override
     * @param httpRequest 请求对象，其中的URL支持占位符，占位符，如/api/v1/user/{id}
     * @return
     */
    default byte[] get(HttpRequest httpRequest) throws IOException {
        String url = this.handleUrlParams(httpRequest.getUrl(), httpRequest.getParams());
        httpRequest.setUrl(url);
        return doGet(httpRequest);
    }

    /**
     * 发起post请求，实现类可选择性Override
     * @param httpRequest 请求对象，其中的URL支持占位符，如/api/v1/user/{id}
     * @return
     */
    default byte[] post(HttpRequest httpRequest) {
        String url = this.handleUrlParams(httpRequest.getUrl(), httpRequest.getParams());
        httpRequest.setUrl(url);
        return doPost(httpRequest);
    }

    /**
     * 发起delete请求，实现类可选择性Override
     * @param httpRequest 请求对象，其中的URL支持占位符，如/api/v1/user/{id}
     * @return
     */
    default byte[] delete(HttpRequest httpRequest) {
        String url = this.handleUrlParams(httpRequest.getUrl(), httpRequest.getParams());
        httpRequest.setUrl(url);
        return doDelete(httpRequest);
    }

    /**
     * 发起put请求，实现类可选择性Override
     * @param httpRequest 请求对象，其中的URL支持占位符，如/api/v1/user/{id}
     * @return
     */
    default byte[] put(HttpRequest httpRequest) {
        String url = this.handleUrlParams(httpRequest.getUrl(), httpRequest.getParams());
        httpRequest.setUrl(url);
        return doPut(httpRequest);
    }

    /**
     * 处理URL参数默认实现，实现类可酌情重载
     * @param url
     * @param params
     * @return
     */
    default String handleUrlParams(final String url, Map params) {
        if(params == null || params.isEmpty() || params.size() <= 0) {
            return url;
        }
        String newUrl = url;
        Matcher m = urlParamPattern.matcher(url);
        List<MachedUrlParam> machedUrlParamList = new ArrayList<>();

        while (m.find()) {
            machedUrlParamList.add(new MachedUrlParam(m.group(), m.start(), m.end()));
        }

        String paramName;
        Object paramVal;
        for(MachedUrlParam machedUrlParam : machedUrlParamList) {
            paramName = machedUrlParam.getParamName();
            paramVal = params.get(paramName);
            newUrl.replace(String.format("%s%s%s", URL_PLACEHOLDER_START, paramName, URL_PLACEHOLDER_END)
                    , paramVal == null ? "" : paramVal.toString());
        }

        return newUrl;
    }

    /**
     * 提取的Url参数对象
     */
    class MachedUrlParam {
        private String paramName;
        private int start;
        private int end;

        public MachedUrlParam(String paramName, int start, int end) {
            this.paramName = paramName;
            this.start = start;
            this.end = end;
        }

        public String getParamName() {
            return paramName;
        }

        public void setParamName(String paramName) {
            this.paramName = paramName;
        }

        public int getStart() {
            return start;
        }

        public void setStart(int start) {
            this.start = start;
        }

        public int getEnd() {
            return end;
        }

        public void setEnd(int end) {
            this.end = end;
        }
    }
}
