package order.web.service;

import cart.api.CartApi;
import cart.api.dto.cart.CartDto;
import cart.api.dto.cart.CartItemDto;
import cms.api.OrganizationApi;
import cms.api.dto.OrganizationDto;
import com.alibaba.fastjson.JSON;
import com.google.common.base.Joiner;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import goods.api.ProductApi;
import goods.dto.product.ProductDto;
import lombok.extern.slf4j.Slf4j;
import ma.glasnost.orika.MapperFacade;
import member.api.AddressCommonApi;
import member.api.CoreCompDepartUserApi;
import member.api.MemberApi;
import member.api.dto.core.CoreCompanyDto;
import member.api.vo.MemberVo;
import order.Contants;
import order.api.OrderApi;
import order.exceptions.IllegalAreaCodeException;
import order.exceptions.temporder.TempOrderCreateException;
import order.vo.GoodsInfoVo;
import order.vo.TempOrderCreateRequest;
import order.vo.TempOrderVo;
import order.vo.response.OrderPrice;
import order.vo.response.OrderStock;
import order.web.config.Constant;
import order.web.vo.*;
import order.web.vo.response.WebOrderPrice;
import order.web.vo.response.WebOrderStock;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;
import sinomall.global.dto.WebResponse;
import store.api.StoreApi;
import store.api.dto.modeldto.core.StoreDto;
import utils.GlobalContants;
import utils.Lang;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author Liang Wenxu
 * @since 2018/7/20
 */
@Component
@Slf4j
public class OrderWebService {
    @MotanReferer
    MemberApi memberApi;

    @MotanReferer
    CoreCompDepartUserApi coreCompDepartUserApi;

    @MotanReferer
    OrderApi orderApi;

    @MotanReferer
    OrganizationApi organizationApi;

    @MotanReferer
    AddressCommonApi addressApi;

    @MotanReferer
    StoreApi storeApi;

    @MotanReferer
    ProductApi productApi;

    @MotanReferer
    CartApi cartApi;

    @Autowired
    MapperFacade orderMapperFacade;

    @Autowired
    StringRedisTemplate redisTemplate;

    /**
     * 查询临时订单的库存信息
     * @param tempOrderVo
     * @return
     */
    public WebResponse<WebOrderStock> getOrderStock(WebTempOrderVo tempOrderVo) {
        WebOrderStock webOrderStock = null;
        try {
            OrderStock orderStock = orderApi.tempOrderStock(orderMapperFacade.map(tempOrderVo, TempOrderVo.class));
            webOrderStock = orderMapperFacade.map(orderStock, WebOrderStock.class);
        } catch (IllegalAreaCodeException e) {
            return WebResponse.ERROR("1001", e.getMessage());
        }
        return WebResponse.SUCCESS(webOrderStock);
    }

    /**
     * 创建临时订单
     * @param webRequest
     * @param memberVo
     * @return
     */
    public WebResponse<WebTempOrderVo> createTempOrder(TempOrderCreateWebRequest webRequest, MemberVo memberVo, HttpSession session) {
        WebResponse<WebTempOrderVo> webResponse = null;

        CoreCompanyDto coreCompany = coreCompDepartUserApi.getBuyerCompany(memberVo.getUser().getId());
        OrganizationDto organization = organizationApi.findOrganizationByOrgCode(webRequest.getOrganizationCode());

        WebTempOrderVo tempOrder = null;

        TempOrderCreateRequest tempOrderCreateRequest = new TempOrderCreateRequest();
        tempOrderCreateRequest.setOrganizationCode(webRequest.getOrganizationCode());
        tempOrderCreateRequest.setOrgainzationId(organization.getId());
        tempOrderCreateRequest.setUserAgent(webRequest.getUserAgent());

        // 处理地区编码
        Map<Integer, String> areaNodeList = addressApi.areaCodeNodeList(webRequest.getAreaCode());
        tempOrderCreateRequest.setProvinceCode(areaNodeList.get(1));
        tempOrderCreateRequest.setCityCode(areaNodeList.get(2));
        tempOrderCreateRequest.setCountyCode(areaNodeList.get(3));
        tempOrderCreateRequest.setTownCode(areaNodeList.get(4));

        // 处理用户信息
        tempOrderCreateRequest.setUserId(memberVo.getUser().getId());
        tempOrderCreateRequest.setMemberId(memberVo.getMember().getId());

        // 单独下单，批量下单另外进行实现
        tempOrderCreateRequest.setOrderMode(Contants.ORDER_MODE_NORMAL);

        // 处理购买商品
        if (webRequest.getProducts() != null && !Lang.isEmpty(webRequest.getProducts())) {
            // 直接从商品生成
            tempOrderCreateRequest.setGoodsInfoVos(requestProduct2GoodsInfoes(webRequest.getProducts()));
        } else if (webRequest.getCartItemIds() != null && !Lang.isEmpty(webRequest.getCartItemIds())) {
            // 从cartItem生成
            tempOrderCreateRequest.setGoodsInfoVos(requestCartItems2GoodsInfoes(webRequest.getCartItemIds(), memberVo.getUser().getId()));
        }

        // 调用Api进行处理
        try {
            TempOrderVo tempOrderRs = orderApi.newTempOrder(tempOrderCreateRequest);
            // 处理用于Web端的VO，组织部分显示用的对象
            tempOrder = orderMapperFacade.map(tempOrderRs, WebTempOrderVo.class);
            // 处理店铺展示信息
            tempOrder.getSubOrders().forEach(it -> {
                StoreDto store = getStore(it.getStoreId());
                it.setStoreTitle(store.getStoreExt().getStoreName());
                it.setStoreShortName(store.getStoreExt().getShortName());
                it.setStoreNeedFeright(!it.getIsFreeShippingFee());
                if(store.getCategory() != null) {
                    it.setStoreCatName(store.getCategory().getName());
                }
                it.setStoreType(store.getStoreExt().getType());
                it.setStorePath(store.getStoreExt().getStorePath());
            });


            tempOrder = addInvoiceToNewTempOrder(tempOrder, coreCompany == null ? null : coreCompany.getId());

            webResponse = WebResponse.SUCCESS(tempOrder);

            addTempOrder2Session(tempOrder, session);
        } catch (TempOrderCreateException e) {
            webResponse = WebResponse.ERROR(e.getCode().getValue(), e.getMessage());
        }

        return webResponse;
    }

    /**
     * 获取临时订单价格接口
     * @param tempOrderVo
     * @return
     */
    public WebResponse<WebOrderPrice> getOrderPrice(WebTempOrderVo tempOrderVo) {
        WebOrderPrice webOrderPrice = null;
        TempOrderVo tempOrderVoApiReq = orderMapperFacade.map(tempOrderVo, TempOrderVo.class);
        Map<Integer, String> areaCodeMap = addressApi.areaCodeNodeList(tempOrderVo.getAreaCode());

        if(areaCodeMap != null) {
            tempOrderVoApiReq.setAreaCodeList(areaCodeMap.entrySet().stream().map(Map.Entry::getValue).collect(Collectors.toList()));
        }

        // 获取地区的nodeList
        try {
            OrderPrice orderPrice = orderApi.tempOrderPrice(tempOrderVoApiReq);
            webOrderPrice = orderMapperFacade.map(orderPrice, WebOrderPrice.class);
        } catch (IllegalAreaCodeException e) {
            return WebResponse.ERROR("1001", e.getMessage());
        }
        return WebResponse.SUCCESS(webOrderPrice);
    }

    /**
     * 将TempOrderProduct转换成GoodsInfoVo
     * @param products
     * @return
     */
    private List<GoodsInfoVo> requestProduct2GoodsInfoes(List<TempOrderProduct> products) {
        //TODO：下面一坨后续优化为缓存获取
        // 提取所有的产品ID
        List<String> productIds = products.stream()
                .map(TempOrderProduct::getProductId).distinct().collect(Collectors.toList());
        // 获取产品
        List<ProductDto> productList = productApi.findByIds(productIds);
        // 提取店铺ID
        List<String> storeIds = productList.stream()
                .map(ProductDto::getStoreId).collect(Collectors.toList());
        // 获取店铺
        List<store.api.dto.modeldto.core.StoreDto> storeDtos = storeApi.findByStoreIds(storeIds);
        // 转换成Map
        Map<String, ProductDto> productMap = productList.stream()
                .collect(Collectors.toMap(ProductDto::getId, Function.identity()));
        Map<String, store.api.dto.modeldto.core.StoreDto> storeMap = storeDtos.stream()
                .collect(Collectors.toMap(store.api.dto.modeldto.core.StoreDto::getId, Function.identity()));
        ProductDto tmpProduct;
        store.api.dto.modeldto.core.StoreDto tmpStore;
        GoodsInfoVo goodsInfoVo;
        List<GoodsInfoVo> goodsInfoVoList = new ArrayList<>();
        for(TempOrderProduct p : products) {
            tmpProduct = productMap.get(p.getProductId());
            tmpStore = storeMap.get(tmpProduct.getStoreId());
            if(tmpProduct != null && tmpStore != null) {
                goodsInfoVo = new GoodsInfoVo();
                goodsInfoVo.setProductType(tmpProduct.getType());
                goodsInfoVo.setCount(p.getCounts());
                goodsInfoVo.setProductFashionId(p.getProductFashionId());
                goodsInfoVo.setGoodsId(p.getGoodsId());
                goodsInfoVo.setGoodsType(Contants.GOODS_TYPE_ENTITY);
                goodsInfoVo.setStoreCode(tmpStore.getStoreExt().getCode());
                goodsInfoVo.setStoreType(tmpStore.getStoreExt().getType());
                goodsInfoVo.setStoreId(tmpStore.getId());

                if(p.getAccessories() != null && !Lang.isEmpty(p.getAccessories())) {
                    goodsInfoVo.setChildren(requestProduct2GoodsInfoes(p.getAccessories()));
                }

                goodsInfoVoList.add(goodsInfoVo);
            }
        }

        return goodsInfoVoList;
    }

    /**
     * 将CartItem转换成GoodsInfoVo
     * @param cartItemIds
     * @return
     */
    private List<GoodsInfoVo> requestCartItems2GoodsInfoes(List<String> cartItemIds, String userId) {
        CartDto userCart = cartApi.getUserCart(userId, false);
        List<GoodsInfoVo> goodsInfoVoList = null;
        if(userCart != null) {
            List<CartItemDto> selectedCartItems = new ArrayList<>();

            // jdk8及以上：
            selectedCartItems = userCart.getCartItems().stream().filter(it ->
                    !it.getIsDelete() && cartItemIds.contains(it.getId())
            ).collect(Collectors.toList());

            // jdk8以下:
//            for (String id : cartItemIds) {
//                for (CartItemDto item : userCart.getCartItems()) {
//                    if (item.getId().equals(id)) {
//                        selectedCartItems.add(item);
//                    }
//                }
//            }

            goodsInfoVoList = requestCartItems2GoodsInfoes(selectedCartItems);
        }

        return goodsInfoVoList;
    }

    /**
     * 将购物车商品转换成GoodsInfo
     * @param cartItemDtoList
     * @return
     */
    private List<GoodsInfoVo> requestCartItems2GoodsInfoes(List<CartItemDto> cartItemDtoList) {
        List<GoodsInfoVo> goodsInfoVoList = null;
        if(cartItemDtoList.size() > 0) {
            goodsInfoVoList = new ArrayList<>();
            //TODO：下面一坨后续优化为缓存获取
            // 提取所有的产品ID
            List<String> productIds = cartItemDtoList.stream()
                    .map(CartItemDto::getProductId).distinct().collect(Collectors.toList());
            // 获取产品
            List<ProductDto> productList = productApi.findByIds(productIds);
            // 提取店铺ID
            List<String> storeIds = productList.stream()
                    .map(ProductDto::getStoreId).collect(Collectors.toList());
            // 获取店铺
            List<store.api.dto.modeldto.core.StoreDto> storeDtos = storeApi.findByStoreIds(storeIds);
            // 转换成Map
            Map<String, ProductDto> productMap = productList.stream()
                    .collect(Collectors.toMap(ProductDto::getId, Function.identity()));
            Map<String, store.api.dto.modeldto.core.StoreDto> storeMap = storeDtos.stream()
                    .collect(Collectors.toMap(store.api.dto.modeldto.core.StoreDto::getId, Function.identity()));
            ProductDto tmpProduct;
            store.api.dto.modeldto.core.StoreDto tmpStore;
            GoodsInfoVo goodsInfoVo;
            for(CartItemDto cartItem : cartItemDtoList) {
                tmpProduct = productMap.get(cartItem.getProductId());

                if(tmpProduct != null) {
                    tmpStore = storeMap.get(tmpProduct.getStoreId());
                    if(tmpStore != null) {
                        goodsInfoVo = new GoodsInfoVo();
                        goodsInfoVo.setStoreId(cartItem.getStoreId());
                        goodsInfoVo.setStoreCode(tmpStore.getStoreExt().getCode());
                        goodsInfoVo.setStoreType(tmpStore.getStoreExt().getType());
                        goodsInfoVo.setCount(cartItem.getCount());
                        goodsInfoVo.setProductType(tmpProduct.getType());
                        goodsInfoVo.setProductFashionId(cartItem.getProductFashId());
                        goodsInfoVo.setGoodsType(Contants.GOODS_TYPE_ENTITY);
                        goodsInfoVo.setGoodsId(cartItem.getGoodsId());
                        if(cartItem.getChildren() != null && !Lang.isEmpty(cartItem.getChildren())) {
                            goodsInfoVo.setChildren(requestCartItems2GoodsInfoes(cartItem.getChildren()));
                        }

                        goodsInfoVoList.add(goodsInfoVo);
                    }
                }
            }
        }
        return goodsInfoVoList;
    }

    /**
     * 获取用户信息
     *
     * @param request
     * @return
     */
    public MemberVo findMemberVo(HttpServletRequest request) {
        String memberId = (String) request.getSession().getAttribute(GlobalContants.SESSION_MEMBER_ID);
        String userId = (String) request.getSession().getAttribute(GlobalContants.SESSION_USER_ID);
//        String memberId="c387779d-d456-4c22-8646-251e22892928";
//        String userId="8e4efd4b-b449-445e-82c5-2c41f91e8444";
        /* 优先使用memberID获取 */
        MemberVo memberVo = null;
        if (StringUtils.isNotBlank(memberId)) {
            log.info("memberId" + memberId);
            memberVo = memberApi.getMemberInfo(memberId);
        } else if (StringUtils.isNotBlank(userId)) {
            log.info("userId" + userId);
            memberVo = memberApi.getMemberInfoByUserId(userId);

        }

        /* 开发用： 当前未登录用户直接获取测试用户数据 */
        //  memberVo = memberVo == null ? memberApi.getMemberInfo("27d23064-9b2a-4bd9-ba63-6174a041e671") : memberVo;

        return memberVo;
    }

    /**
     * 添加发票信息到临时订单
     *
     * @param tempOrder
     * @param coreCompanyId
     */
    public WebTempOrderVo addInvoiceToNewTempOrder(WebTempOrderVo tempOrder, String coreCompanyId) {
        List<WebTempInvoiceItemVo> invoiceItemVos = new Vector<>();
        for (WebTempOrderItemVo orderItemVo : tempOrder.getOrderItems()) {
            WebTempInvoiceItemVo item = new WebTempInvoiceItemVo();
            item.setCount(orderItemVo.getCount());
            item.setProductFashionId(orderItemVo.getProductFashionId());
            item.setGoodsId(orderItemVo.getGoodsId());
            invoiceItemVos.add(item);
            if(!Lang.isEmpty(orderItemVo.getChildren())){
                orderItemVo.getChildren().stream().forEach(chl->{
                    WebTempInvoiceItemVo itemChild = new WebTempInvoiceItemVo();
                    itemChild.setCount(orderItemVo.getCount()*chl.getCount());
                    itemChild.setProductFashionId(chl.getProductFashionId());
                    invoiceItemVos.add(itemChild);
                });
            }
        }
        WebTempInvoiceVo invoiceVo = new WebTempInvoiceVo();
        invoiceVo.setTempInvoiceItemVos(invoiceItemVos);
        invoiceVo.setCompanyId(coreCompanyId);
        invoiceVo.setInvoiceType("02");
        tempOrder.setTempInvoiceVos(Arrays.asList(invoiceVo));

        return tempOrder;
    }

    @Cacheable(cacheNames = {Constant.CACHE_NAMES_API_STORE}, key = "#storeId")
    public StoreDto getStore(String storeId) {
        return storeApi.getStoreInfo(storeId);
    }

    /**
     * 保存临时订单到Session
     * @param tempOrder
     * @param session
     */
    public void addTempOrder2Session(WebTempOrderVo tempOrder, HttpSession session) {
        session.setAttribute(
                GlobalContants.TEMP_ORDER_SESSION_KEY_PREFIX + tempOrder.getTmpOrderNo(),
                JSON.toJSONString(tempOrder));
        List<String> tmpIdList = getSessionTmpOrderIds(session);
        tmpIdList.add(tempOrder.getTmpOrderNo());
        putSessionTmpOrderIds(tmpIdList, session);
    }

    /**
     * 判断临时订单ID是否在Session中
     *
     * @param tmpId
     * @param session
     * @return
     */
    public Boolean tmpOrderIdExists(String tmpId, HttpSession session) {
        List<String> tmpIdList = getSessionTmpOrderIds(session);
        return tmpIdList.contains(tmpId);
    }

    /**
     * 从Session中移除临时订单ID(提交后的订单)
     *
     * @param tmpId
     * @param session
     */
    public void removeSessionTmpOrderId(final String tmpId, HttpSession session) {
        List<String> tmpIdList = getSessionTmpOrderIds(session);
        // JDK 8+:
        tmpIdList.removeIf(p -> p.equals(tmpId));
        putSessionTmpOrderIds(tmpIdList, session);

        String s = (String) session.getAttribute(GlobalContants.TEMP_ORDER_SESSION_KEY_PREFIX + tmpId);
        if (!Lang.isEmpty(s)) {
            session.removeAttribute(GlobalContants.TEMP_ORDER_SESSION_KEY_PREFIX + tmpId);
        }
    }

    /**
     * 从Session中获取临时订单列表
     *
     * @param session
     * @return
     */
    public List<String> getSessionTmpOrderIds(HttpSession session) {
        String tmpIdsJoined = (String) session.getAttribute(GlobalContants.ORDER_TEMP_LIST_ID);
        List<String> tmpIdList = new ArrayList<>();
        if (!Lang.isEmpty(tmpIdsJoined)) {
            tmpIdList.addAll(Arrays.asList(tmpIdsJoined.split(",")));
        } else {
            tmpIdList = new ArrayList<>();
        }

        return tmpIdList;
    }

    /**
     * 将临时订单列表放入Session
     *
     * @param tmpIdList
     * @param session
     */
    public void putSessionTmpOrderIds(List<String> tmpIdList, HttpSession session) {
        if (tmpIdList != null && !Lang.isEmpty(tmpIdList)) {
            session.setAttribute(GlobalContants.ORDER_TEMP_LIST_ID, Joiner.on(",").skipNulls().join(tmpIdList));
        }
    }

    /**
     * 获取临时订单
     * @param tempOrderNo
     * @param session
     * @return
     */
    public WebTempOrderVo getTempOrder(String tempOrderNo, HttpSession session) {
        String json = (String) session.getAttribute(
                GlobalContants.TEMP_ORDER_SESSION_KEY_PREFIX + tempOrderNo);
        if (Lang.isEmpty(json)) {
            return null;
        } else {
            return JSON.parseObject(json, WebTempOrderVo.class);
        }
    }
}
