package spider.service.service;

import com.alibaba.fastjson.JSON;
import org.apache.http.HttpEntity;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import spider.model.SpiderLogs;
import spider.model.repository.SpiderLogsRepos;
import spider.service.factory.PriceSpiderHandlerFactory;
import spider.service.handler.PriceSpiderHandler;
import spider.service.utils.SpiderFactoryUtil;
import utils.Lang;
import utils.log.Log;
import utils.log.Logs;
import utils.rpc.motan.ApiResponseVo;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.io.IOException;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.*;

/**
 * @author Liang Wenxu
 * @since 2017-07-04
 * Created by Liang Wenxu on 2017-07-04.
 */
@Component
@Transactional(readOnly = true)
public class SpiderLogsService {
    Log log = Logs.getLog("SpiderLogsService");
    /** Attributes ============== */

    /** Methods ============== */

    /**
     * 分页查询
     * @param storeId 商户ID
     * @param productCode 产品代码
     * @param createTimeStart 日志记录时间区间开始
     * @param createTimeEnd 日志记录时间区间结束
     * @param pageable 分页对象
     * @return 分页结果集
     */
    public Page<SpiderLogs> findSpiderLogs(String storeId, String productCode, Date createTimeStart, Date createTimeEnd, Pageable pageable) {
        return spiderLogsRepos.findAll(new Specification<SpiderLogs>() {
            @Override
            public Predicate toPredicate(Root<SpiderLogs> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
                List<Predicate> where = new ArrayList<>();

                if(!Lang.isEmpty(storeId)) {
                    where.add(cb.equal(root.get("storeId"), storeId));
                }

                if(!Lang.isEmpty(productCode)) {
                    where.add(cb.equal(root.get("productCode"), productCode));
                }

                if(createTimeStart != null) {
                    where.add(cb.greaterThanOrEqualTo(root.get("dateCreated"), createTimeStart));
                }

                if(createTimeEnd != null) {
                    where.add(cb.lessThan(root.get("dateCreated"), createTimeEnd));
                }

                return cb.and(where.toArray(new Predicate[0]));
            }
        }, pageable);
    }

    /**
     * 保存日志
     * @param log
     */
    @Transactional(readOnly = false)
    public void saveLog(SpiderLogs log) {
        spiderLogsRepos.saveAndFlush(log);
    }

    /**
     * 刷新数据库所有商品的价格
     */
    public void doUpdatePrices(Integer poolSize, Integer pageSize, Integer dbPoolSize, Integer dbPageSize) {
        // 查询所有的sku和price
        Date startTime = new Date();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        log.info("doUpdatePrices starts: "+sdf.format(new Date()));
        poolSize = poolSize == null ? 10 : poolSize;
        ExecutorService dbQueryThreadPool = Executors.newFixedThreadPool(dbPoolSize);
        ExecutorService reqThreadPool = Executors.newFixedThreadPool(poolSize);
        Integer totalCount =  jdbcTemplate.queryForObject("SELECT COUNT(0) FROM goods g JOIN store s ON s.ID = g.STORE_ID JOIN STORE_EXT se on se.STORE_ID = s.id WHERE se.CODE='jd'", Integer.class);

        int totalPageNum = (totalCount  +  dbPageSize  - 1) / dbPageSize;
        List<Future<Integer>> dbFutures = new ArrayList<>();

        for(int pageNum=0; pageNum < totalPageNum; pageNum ++) {
            int finalPageNum = pageNum;
            dbFutures.add(
                    dbQueryThreadPool.submit(new Callable<Integer>() {
                        @Override
                        public Integer call() throws Exception {
                            Vector<Future<Integer>> futures = new Vector<>();

                            Pageable pageable = new PageRequest(finalPageNum, dbPageSize);

                            List<Map<String, Object>> dbList = jdbcTemplate.queryForList("SELECT * FROM (SELECT g.sku, g.cost_price, ROWNUM n FROM goods g JOIN store s ON s.ID = g.STORE_ID JOIN STORE_EXT se on se.STORE_ID = s.id WHERE se.CODE='jd') t WHERE t.n > ? AND t.n <= ?",
                                    pageable.getOffset(), pageable.getOffset() + pageable.getPageSize()
                            );
                            log.info("已查询数据" + finalPageNum + "/" + totalPageNum);
                            // 再进行分页，进行商品的查询
                            int totalDbCounts = dbList.size();
                            int totalDbPages = (totalDbCounts  +  pageSize  - 1) / pageSize;
                            int pageNum = 1;

                            while (pageNum <= totalDbPages) {
                                int finalPageNum1 = pageNum;
                                Future<Integer> future = reqThreadPool.submit(new Callable<Integer>() {
                                    @Override
                                    public Integer call() throws Exception {
                                        int offset = (finalPageNum1 - 1) * pageSize;
                                        int max = offset + pageSize - 1;
                                        List<Map<String, Object>> datas = dbList.subList(offset, max > dbList.size() ? dbList.size() - 1 : max);
                                        List<String> skus = Lang.grepList(datas, "sku", "");
                                        String priceUrl = "http://p.3.cn:9999/prices/mgets?skuIds={sku}";

                                        StringBuilder skuIdSb = new StringBuilder();
                                        for(String sku : skus) {
                                            skuIdSb.append(",").append("J_").append(sku);
                                        }

                                        priceUrl = priceUrl.replace("{sku}", skuIdSb.substring(1));

                                        RequestConfig requestConfig = RequestConfig.custom()
                                                .setConnectTimeout(500).setConnectionRequestTimeout(500)
                                                .setSocketTimeout(500).build();

                                        CloseableHttpClient httpclient = HttpClients.createDefault();
                                        HttpGet httpGet = new HttpGet(priceUrl);
                                        httpGet.setConfig(requestConfig);
                                        CloseableHttpResponse response = null;

                                        Map<String, SpiderLogs> returnData = new HashMap<>();

                                        try {
                                            response = httpclient.execute(httpGet);
                                            System.out.println(response.getStatusLine());
                                            if(response.getStatusLine().getStatusCode() == 200) {
                                                HttpEntity entity = response.getEntity();
                                                String contentJson = EntityUtils.toString(entity);
//                                                System.out.println("JD Spider HTTP Response: " + contentJson);
                                                boolean sucFlag = false;
                                                try {
                                                    List<Map> jsonArray = JSON.parseArray(contentJson, Map.class);
                                                    Date startTime = new Date();
                                                    for(Map m : jsonArray) {
                                                        String sku = ((String) m.get("id")).toUpperCase().replace("J_", "");

                                                        jdbcTemplate.update("INSERT INTO TEMP_PRICE_SYIC_LOG(SKU, cost_price, trd_price, used_time, sync_time) VALUES (?, ?, ?, ?, ?)",
                                                                sku,
                                                                Lang.findInBeanList(datas, "sku", sku).get("cost_price"),
                                                                Lang.isEmpty(m.get("p")) ? BigDecimal.ZERO : new BigDecimal(m.get("p").toString()),
                                                                (new Date()).getTime() - startTime.getTime(),
                                                                new Date()
                                                        );
                                                    }

                                                    log.info("已处理"+ finalPageNum1 +"/"+totalDbPages);
                                                } catch (Exception e) {
                                                    e.printStackTrace();
                                                }
                                            }
                                        } catch (ClientProtocolException e) {
                                            e.printStackTrace();
                                        } catch (IOException e) {
                                            e.printStackTrace();
                                        } finally {
                                            if(response != null) {
                                                try {
                                                    response.close();
                                                } catch (IOException e) {
                                                    e.printStackTrace();
                                                }
                                            }
                                        }


                                        return datas.size();
                                    }
                                });
                                futures.add(future);
                                pageNum ++;
                            }
                            int handledCounts = 0;
                            for(Future<Integer> f : futures) {
                                try {
                                    handledCounts += f.get();
                                } catch (InterruptedException e) {
                                    e.printStackTrace();
                                } catch (ExecutionException e) {
                                    e.printStackTrace();
                                }
                            }

                            log.info("已完成处理" + finalPageNum + "/" + totalPageNum+", 共处理"+handledCounts+"条数据");
                            return handledCounts;
                        }
                    })
            );
        }

        int totalHandled = 0;
        for(Future<Integer> f : dbFutures) {
            try {
                totalHandled += f.get();
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (ExecutionException e) {
                e.printStackTrace();
            }
        }

        log.info("所有京东商品价格获取完毕，总处理数据" + totalHandled + "，总耗时:" + ((new Date()).getTime() - startTime.getTime()) / 1000 + "s");

        dbQueryThreadPool.shutdown();
        reqThreadPool.shutdown();
    }

    /** Dependent Components ============== */
    @Autowired
    SpiderFactoryUtil spiderFactoryUtil;

    @Autowired
    SpiderLogsRepos spiderLogsRepos;

    @Autowired
    JdbcTemplate jdbcTemplate;
    /** Setter and getters ============== */
}
