package order.service.events.listener;

import cart.api.dto.order.OrderJmsMessage;
import cart.api.dto.order.OrderMainDto;
import order.model.po.OrderMain;
import ma.glasnost.orika.MapperFacade;
import order.service.events.OrderEvent;
import order.service.events.OrderStatusChangeEvent;
import order.service.jms.JmsHelper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.event.SmartApplicationListener;
import org.springframework.stereotype.Component;
import store.api.dto.modeldto.core.StoreDto;
import utils.log.Log;
import utils.log.Logs;

/**
 *
 * 协议采购用订单状态改变监听器
 * 当OrderMain状态进行改变时，同步改变到协议订单表
 * @author Liang Wenxu
 * @since 2018-01-12
 */
@Component
public class ProtocolOrderStatusChangeListener implements SmartApplicationListener {
    Log log = Logs.getLog(ProtocolOrderStatusChangeListener.class.getName());
    private static final String orgCode = "protocol";

    @Value("${activemq.queue.orderEvents:}")
    private String orderEventQueue;

    @Autowired
    JmsHelper jmsHelper;
    @Autowired
    MapperFacade mapperFacade;

    @Override
    public boolean supportsEventType(Class<? extends ApplicationEvent> eventType) {
        return eventType == OrderStatusChangeEvent.class;
    }

    @Override
    public boolean supportsSourceType(Class<?> sourceType) {
        return sourceType == OrderMain.class;
    }

    @Override
    public void onApplicationEvent(ApplicationEvent event) {
        OrderStatusChangeEvent orderEvent = (OrderStatusChangeEvent) event;
        OrderMain orderMain = (OrderMain) orderEvent.getSource();
        OrderMainDto orderMainDto = mapperFacade.map(orderMain, OrderMainDto.class);
        if (StoreDto.STORE_TYPE_PROTOCOL.equals(orderMain.getStoreType())) {
            log.info("orderNo: {0}, orgCode: {1} ,协议采购订单状态改变事件", orderMain.getOrderNo() , ((OrderEvent) event).getOrgCode());
            // 发送队列消息
            jmsHelper.sendQueuedMessage(OrderJmsMessage.newMessage(
                    orderMainDto, OrderJmsMessage.OrderEvents.STATUS_CHANGED), orderEventQueue);
//            protocolApi
        } else {
            log.info("orderNo: {0}, orgCode: {1} ,非协议采购订单订单状态改变事件", orderMain.getOrderNo() , ((OrderEvent) event).getOrgCode());
        }

    }

    @Override
    public int getOrder() {
        return 0;
    }
}
