package order.service.operator.impl;

import goods.vo.SalesPriceCalVo;
import lombok.extern.slf4j.Slf4j;
import order.exceptions.IllegalAreaCodeException;
import order.service.operator.OrderFreightOperator;
import order.service.operator.OrderPriceOperator;
import order.service.stubs.OutsideProductApiStub;
import order.service.stubs.ProductFashionApiStub;
import order.service.stubs.SalesPriceRateApiStub;
import order.vo.TempOrderItemVo;
import order.vo.TempOrderVo;
import order.vo.TempSubOrderVo;
import order.vo.response.OrderItemPrice;
import order.vo.response.OrderPrice;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import outsideapi.vo.FashionPriceParamVo;
import outsideapi.vo.FashionPriceRequestVo;
import outsideapi.vo.FashionPriceVo;
import outsideapi.vo.HandlerRespVo;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.*;
import java.util.stream.Collectors;

/**
 * 默认订单价格操作者
 * <p>如要覆盖所有的操作者实现，编写新的实现类，将本类上的Component注解去掉，新实现类的注解BeanName保持为orderPriceOperator</p>
 * <p>如要扩展成多实现的，可以通过注册不同的beanName（使用Component的value属性指定），然后在注入时根据名字注入如OrderPriceOperator newPriceOperator;</p>
 * @author Liang Wenxu
 * @since 2018/7/26
 */
@Component("orderPriceOperator")
@Slf4j
public class DefaultOrderPriceOperator implements OrderPriceOperator {

    @Autowired
    ProductFashionApiStub productFashionApi;

    @Autowired
    OutsideProductApiStub outsideProductApi;

    @Autowired
    SalesPriceRateApiStub salesPriceRateApi;

    @Autowired
    OrderFreightOperator orderFreightOperator;

    @Override
    public OrderPrice get(TempOrderVo tempOrderVo) throws IllegalAreaCodeException {
        // 获取商品价格
        FashionPriceRequestVo fashionPriceRequestVo = new FashionPriceRequestVo(tempOrderVo.getAreaCodeList());
        FashionPriceParamVo fashionPriceParamVo;
        ExecutorService pool = new ThreadPoolExecutor(0, 3,
                10L, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<Runnable>());

        // 需要按照店铺进行分组调用
        Map<String, List<TempOrderItemVo>> storeGroups = tempOrderVo.getOrderItems()
            .stream().collect(Collectors.groupingBy(TempOrderItemVo::getStoreId));
        List<Future<HandlerRespVo<List<FashionPriceVo>>>> outsidePriceFutures = new ArrayList<>();
        for(Map.Entry<String, List<TempOrderItemVo>> entry : storeGroups.entrySet()) {
            fashionPriceRequestVo.setFashionPriceParamVos(new ArrayList<>());
            for(TempOrderItemVo itemVo : entry.getValue()) {
                fashionPriceParamVo = new FashionPriceParamVo();
                fashionPriceParamVo.setProductCode(itemVo.getProductCode());
                fashionPriceParamVo.setFactionId(itemVo.getProductFashionId());
                fashionPriceRequestVo.getFashionPriceParamVos().add(fashionPriceParamVo);
            }
            // 异步方式调用API提高效率
            outsidePriceFutures.add(pool.submit(() -> {
                long queryFashionPriceTime = System.currentTimeMillis();
                HandlerRespVo<List<FashionPriceVo>> outsideApiRes = outsideProductApi.queryFashionPrice(entry.getKey(), fashionPriceRequestVo);
                log.info("第三方实时价格查询（单个线程）耗时 {} ms", System.currentTimeMillis() - queryFashionPriceTime);
                return outsideApiRes;
            }));
        }
        // 别忘了关闭线程池噢
        pool.shutdown();

        long queryFashionPriceTime = System.currentTimeMillis();
        List<FashionPriceVo> allFashionPrices = new ArrayList<>();

        for(Future<HandlerRespVo<List<FashionPriceVo>>> f : outsidePriceFutures) {
            try {
                HandlerRespVo<List<FashionPriceVo>> frs = f.get();
                if(HandlerRespVo.RESPONSE_STATUS_SUCCESS.equals(frs.getStatus())) {
                    if(frs.getData() != null && frs.getData().size() > 0) {
                        allFashionPrices.addAll(frs.getData());
                    }

                }
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (ExecutionException e) {
                e.printStackTrace();
            }
        }

        // 处理费率价
        List<SalesPriceCalVo> batchSalesPriceCalList = new ArrayList<>();
        for(FashionPriceVo fp : allFashionPrices) {
            SalesPriceCalVo salesPriceCalVo = new SalesPriceCalVo();
            tempOrderVo.getOrderItems().stream().filter(it -> it.getProductFashionId().equals(fp.getFashionId())).findFirst().ifPresent(oi -> {
                batchSalesPriceCalList.add(new SalesPriceCalVo(fp.getFashionId(), oi.getStoreId(), oi.getStoreCode()
                        , tempOrderVo.getRealOrganizationId(), fp.getSalePrice(), fp.getCostPrice(), fp.getMarketPrice()
                        , fp.getSalePrice(), oi.getCount(), tempOrderVo.getOrganizationCode()));

            });
        }
        long queryOrgSalesPriceTime = System.currentTimeMillis();
        List<Map> rateCalRs = salesPriceRateApi.calBatchSalesPrice(batchSalesPriceCalList);
        log.info("查询费率价格耗时 {} ms", System.currentTimeMillis() - queryFashionPriceTime);

        for(FashionPriceVo fp : allFashionPrices) {
            rateCalRs.stream().filter(it -> fp.getFashionId().equals(it.get("fashionId"))).findFirst().ifPresent(it -> {
                fp.setSalePrice((BigDecimal) it.get("salePrice"));
            });
        }

        // 返回结果转换成OrderPrice
        OrderPrice orderPrice = new OrderPrice();
        orderPrice.setOrderItemPrices(new ArrayList<>());
        orderPrice.setFreight(tempOrderVo.getSumShippingFee());
        orderPrice.setOrderNo(tempOrderVo.getTmpOrderNo());
        OrderItemPrice oip;
        FashionPriceVo r;
        for (TempOrderItemVo oi : tempOrderVo.getOrderItems()) {
            oip = new OrderItemPrice(oi.getCount(), oi.getProductFashionId());
            Optional<FashionPriceVo> optional = allFashionPrices.stream().filter(i -> i.getFashionId().equals(oi.getProductFashionId())).findFirst();
            if(optional.isPresent()) {
                r = optional.get();
                genOrderItemPrice(oip, r);

            } else {
                genOrderItemPrice(oip, oi);
            }
            orderPrice.setSubOrderPrices(makeOrderPrice(tempOrderVo.getSubOrders(), allFashionPrices));
            orderPrice.getOrderItemPrices().add(oip);
//            orderPrice.setFreight(tempOrderVo.getSumShippingFee());
        }

        // 计算总价
        orderPrice.recalSumPrices();
        log.info("第三方实时价格查询（总）耗时 {} ms", System.currentTimeMillis() - queryFashionPriceTime);

        return orderPrice;
    }

    /**
     * 根据fashionPriceVo生成OrderItemPrice
     * @param fashionPrice
     * @return
     */
    private OrderItemPrice genOrderItemPrice(OrderItemPrice oip, FashionPriceVo fashionPrice) {
        oip.setCostPrice(fashionPrice.getCostPrice());
        oip.setMarketPrice(fashionPrice.getMarketPrice());
        oip.setSalesPrice(fashionPrice.getSalePrice());
        oip.setSumCostPrice(oip.getCostPrice().multiply(BigDecimal.valueOf(oip.getCount())));
        oip.setSumSalesPrice(oip.getSalesPrice().multiply(BigDecimal.valueOf(oip.getCount())));
        oip.setSumMarketPrice(oip.getMarketPrice().multiply(BigDecimal.valueOf(oip.getCount())));
        return oip;
    }

    /**
     * 根据TempOrderItemVo生成OrderItemPrice
     * @param tempOrderItem
     * @return
     */
    private OrderItemPrice genOrderItemPrice(OrderItemPrice oip, TempOrderItemVo tempOrderItem) {
        FashionPriceVo f = new FashionPriceVo();
        f.setCostPrice(tempOrderItem.getCostPrice());
        f.setMarketPrice(tempOrderItem.getMarketPrice());
        f.setSalePrice(tempOrderItem.getSalePrice());
        return genOrderItemPrice(oip, f);
    }

    /**
     * 根据临时子订单获取订单金额
     *
     * @param subOrder 子订单，使用此方法时需确认子订单下再无不同店铺的子单
     * @return
     * @throws IllegalAreaCodeException
     */
    @Override
    public OrderPrice get(TempSubOrderVo subOrder, List<String> areaNodes) throws IllegalAreaCodeException {
        OrderPrice orderPrice = null;

        // 获取商品价格
        FashionPriceRequestVo fashionPriceRequestVo = new FashionPriceRequestVo(areaNodes);
        FashionPriceParamVo fashionPriceParamVo;

        // 需要按照店铺进行分组调用
        fashionPriceRequestVo.setFashionPriceParamVos(new ArrayList<>());
        for(TempOrderItemVo itemVo : subOrder.getOrderItems()) {
            fashionPriceParamVo = new FashionPriceParamVo();
            fashionPriceParamVo.setProductCode(itemVo.getProductCode());
            fashionPriceParamVo.setFactionId(itemVo.getProductFashionId());
            fashionPriceRequestVo.getFashionPriceParamVos().add(fashionPriceParamVo);
        }

        HandlerRespVo<List<FashionPriceVo>> outsideApiRes = outsideProductApi.queryFashionPrice(subOrder.getStoreId(), fashionPriceRequestVo);

        List<FashionPriceVo> results = null;
        if(HandlerRespVo.RESPONSE_STATUS_SUCCESS.equals(outsideApiRes.getStatus())) {
            orderPrice = new OrderPrice();
            results = outsideApiRes.getData();
            orderPrice.setFreight(subOrder.getSumShippingFee());
            // 返回结果转换成OrderPrice
            orderPrice.setOrderItemPrices(new ArrayList<>());
            OrderItemPrice oip;
            FashionPriceVo r;
            for (TempOrderItemVo oi : subOrder.getOrderItems()) {
                oip = new OrderItemPrice(oi.getCount(), oi.getProductFashionId());
                Optional<FashionPriceVo> optional = results.stream().filter(i -> i.getFashionId().equals(oi.getProductFashionId())).findFirst();
                if(optional.isPresent()) {
                    r = optional.get();
                    genOrderItemPrice(oip, r);
                } else {
                    genOrderItemPrice(oip, oi);
                }
                orderPrice.getOrderItemPrices().add(oip);
            }

            // 计算总价
            orderPrice.recalSumPrices();
        }

        return orderPrice;
    }

    /**
     * 根据子订单生成订单金额返回
     * @param subOrderList
     * @param allItemPrice
     * @return
     */
    private List<OrderPrice> makeOrderPrice(List<TempSubOrderVo> subOrderList, final List<FashionPriceVo> allItemPrice) {
        OrderItemPrice oip;
        OrderPrice orderPrice;
        FashionPriceVo r;
        List<OrderPrice> subOrderPriceList = new ArrayList<>();
        for(TempSubOrderVo subOrderVo : subOrderList) {
            orderPrice = new OrderPrice();
            orderPrice.setOrderNo(subOrderVo.getTmpOrderNo());
            orderPrice.setOrderItemPrices(new ArrayList<>());
            orderPrice.setFreight(subOrderVo.getSumShippingFee());
            for(TempOrderItemVo oi : subOrderVo.getOrderItems()) {
                oip = new OrderItemPrice(oi.getCount(), oi.getProductFashionId());
                Optional<FashionPriceVo> optional = allItemPrice.stream().filter(i -> i.getFashionId().equals(oi.getProductFashionId())).findFirst();
                if(optional.isPresent()) {
                    r = optional.get();
                    genOrderItemPrice(oip, r);
                } else {
                    genOrderItemPrice(oip, oi);
                }
                orderPrice.getOrderItemPrices().add(oip);
            }
            if(subOrderVo.getSubOrders() != null && subOrderVo.getSubOrders().size() > 0) {
                orderPrice.setSubOrderPrices(makeOrderPrice(subOrderVo.getSubOrders(), allItemPrice));
            }

//            orderPrice.setFreight(subOrderVo.getSumShippingFee());
            // 计算总价
            orderPrice.recalSumPrices();
            subOrderPriceList.add(orderPrice);
        }

        return subOrderPriceList;
    }
}
