package order.service.operator.impl;

import goods.vo.FashionIdAndCountVo;
import goods.vo.FashionStock;
import goods.vo.FashionStockRequest;
import goods.vo.FashionStockResult;
import lombok.extern.slf4j.Slf4j;
import order.exceptions.IllegalAreaCodeException;
import order.service.operator.OrderStockOperator;
import order.service.stubs.ProductFashionApiStub;
import order.vo.TempOrderItemVo;
import order.vo.TempOrderVo;
import order.vo.TempSubOrderVo;
import order.vo.response.OrderItemStock;
import order.vo.response.OrderStock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import sinomall.global.common.response.BaseResponse;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

/**
 * 默认订单库存操作者
 * <p>如要覆盖所有的操作者实现，编写新的实现类，将本类上的Component注解去掉，新实现类的注解BeanName保持为orderStockOperator</p>
 * <p>如要扩展成多实现的，可以通过注册不同的beanName（使用Component的value属性指定），然后在注入时根据名字注入如OrderStockOperator newOperator;</p>
 * @author Liang Wenxu
 * @since 2018/7/26
 */
@Component("orderStockOperator")
@Slf4j
public class DefaultOrderStockOperator implements OrderStockOperator {

    @Autowired
    ProductFashionApiStub productFashionApi;

    @Override
    public OrderStock get(TempOrderVo tempOrderVo) throws IllegalAreaCodeException {
        List<FashionIdAndCountVo> fashionDetailVoList = new ArrayList<>();
        tempOrderVo.getOrderItems().forEach(it -> {
            fashionDetailVoList.add(
                    new FashionIdAndCountVo(it.getProductFashionId(), it.getCount()));
        });


        /** 查询库存状态 */
        OrderStock result = new OrderStock();
        List<OrderItemStock> mainItemStocks = new ArrayList<>();
        FashionStockRequest fashionStockRequest = new FashionStockRequest();
        fashionStockRequest.setAreaCode(tempOrderVo.getAreaCode());
        fashionStockRequest.setOrgId(tempOrderVo.getRealOrganizationId());
        fashionStockRequest.setFashionIdAndCountVoList(fashionDetailVoList);

        BaseResponse<FashionStockResult> apiResponse = productFashionApi.queryFashionStockStatus(fashionStockRequest);
        boolean noStockFlag = true; // 包含无库存状态商品标志
        if(apiResponse.isSuccess()) {
            noStockFlag = noStockFlag && apiResponse.getResult().getNoStockFlag();
            // 处理订单商品详情
            for(FashionStock r : apiResponse.getResult().getDetails()) {
                mainItemStocks.add(genByFashionStock(r));
            }

        } else {
            if("1001".equals(apiResponse.getResultCode()) || "1002".equals(apiResponse.getResultCode())) {
                throw new IllegalAreaCodeException(apiResponse.getResultMessage());
            }
            noStockFlag = noStockFlag && false;
        }

        result.setItemStocks(mainItemStocks);
        result.setHasNoStock(noStockFlag);


        // 根据TempOrderVo，处理其子单
        if(tempOrderVo.getSubOrders().size() > 1) {
            result.setSubOrderStock(new ArrayList<>());
            for(TempSubOrderVo o : tempOrderVo.getSubOrders()) {
                result.getSubOrderStock().add(makeOrderStock(o, mainItemStocks));
            }
        }

        return result;
    }

    /**
     * 获取订单库存
     *
     * @param tempOrderVo 临时订单VO，将生成的实例整体怼过来，别瞎改里面的东西
     * @param areaCode
     * @param organizationalId
     * @return 返回订单的库存装状态，父子结构与tempOrder中将保持一致，itemStocks保存商品的库存详情，如为父订单则会包括所有子单的商品
     */
    @Override
    public OrderStock get(TempSubOrderVo tempOrderVo, String areaCode, String organizationalId) throws IllegalAreaCodeException {
        List<FashionIdAndCountVo> fashionDetailVoList = new ArrayList<>();
        tempOrderVo.getOrderItems().forEach(it -> {
            fashionDetailVoList.add(
                    new FashionIdAndCountVo(it.getProductFashionId(), it.getCount()));
        });


        /** 查询库存状态 */
        OrderStock result = new OrderStock();
        List<OrderItemStock> mainItemStocks = new ArrayList<>();
        FashionStockRequest fashionStockRequest = new FashionStockRequest();
        fashionStockRequest.setAreaCode(areaCode);
        fashionStockRequest.setOrgId(organizationalId);
        fashionStockRequest.setFashionIdAndCountVoList(fashionDetailVoList);

        BaseResponse<FashionStockResult> apiResponse = productFashionApi.queryFashionStockStatus(fashionStockRequest);
        boolean noStockFlag = true; // 包含无库存状态商品标志
        if(apiResponse.isSuccess()) {
            noStockFlag = noStockFlag && apiResponse.getResult().getNoStockFlag();
            // 处理订单商品详情
            for(FashionStock r : apiResponse.getResult().getDetails()) {
                mainItemStocks.add(genByFashionStock(r));
            }
        } else {
            if("1001".equals(apiResponse.getResultCode()) || "1002".equals(apiResponse.getResultCode())) {
                throw new IllegalAreaCodeException(apiResponse.getResultMessage());
            }
            noStockFlag = noStockFlag && false;
        }

        result.setItemStocks(mainItemStocks);
        result.setHasNoStock(noStockFlag);


        // 根据TempOrderVo，处理其子单
        if(tempOrderVo.getSubOrders() != null && tempOrderVo.getSubOrders().size() > 1) {
            result.setSubOrderStock(new ArrayList<>());
            for(TempSubOrderVo o : tempOrderVo.getSubOrders()) {
                result.getSubOrderStock().add(makeOrderStock(o, mainItemStocks));
            }
        }

        return result;
    }

    /**
     * 根据原临时订单的父子结构处理库存详情
     * @param tempSubOrderVo
     * @param orderItemStocks
     * @return
     */
    private OrderStock makeOrderStock(TempSubOrderVo tempSubOrderVo, final List<OrderItemStock> orderItemStocks) {
        OrderStock orderStock = new OrderStock();
        orderStock.setItemStocks(new ArrayList<>());
        orderStock.setSubOrderStock(new ArrayList<>());

        // 将包含在子单的OrderItemStock放入各自子单中
        List<String> orderItemFashionIds = tempSubOrderVo.getOrderItems().stream().map(TempOrderItemVo::getProductFashionId).collect(Collectors.toList());
        List<OrderItemStock> selected =  orderItemStocks.stream().filter(ois ->
            orderItemFashionIds.contains(ois.getFashionId())
        ).collect(Collectors.toList());

        for(OrderItemStock ois : selected) {
            orderStock.getItemStocks().add(ois);
        }

        // 处理子单的有货状态
        AtomicBoolean hasStock = new AtomicBoolean(true);
        orderStock.getItemStocks().forEach(is -> {
//            hasStock.set(hasStock.get() && (is.getOnSale()
//                    && !is.getIsAreaRestrict()
//                    && !OrderItemStock.STOCK_FLAG_NO_STOCK.equals(is.getStockFlag())));

            hasStock.set(hasStock.get() && (is.getOnSale()
                    && !OrderItemStock.STOCK_FLAG_NO_STOCK.equals(is.getStockFlag())));
        });

        orderStock.setHasNoStock(!hasStock.get());

        // 子单的子单，无穷尽也~
        if(tempSubOrderVo.getSubOrders() != null && tempSubOrderVo.getSubOrders().size() > 0) {
            for(TempSubOrderVo so : tempSubOrderVo.getSubOrders()) {
                orderStock.getSubOrderStock().add(makeOrderStock(so, orderItemStocks));
            }
        }


        return orderStock;
    }

    /**
     * 根据供应商接口返回库存生成返回结果
     * @param fashionStock
     * @return
     */
    private OrderItemStock genByFashionStock(FashionStock fashionStock) {
        OrderItemStock itemStock = new OrderItemStock();
        itemStock.setFashionId(fashionStock.getFashionId());
        itemStock.setStockFlag(fashionStock.getStockFlag());
        itemStock.setStockCount(fashionStock.getStockCount());

        if(FashionStock.UP_DOWN_FLAG_DOWN == fashionStock.getUpdownflag()) {
            itemStock.setOnSale(false);
            itemStock.setStockFlag(OrderItemStock.STOCK_FLAG_NO_STOCK);
            itemStock.setNotsaleReason(OrderItemStock.NOTSALE_REASON_PRODUCT_DOWN);
        } else {
            itemStock.setOnSale(fashionStock.getOnSale());
            if(!fashionStock.getOnSale()) {
                itemStock.setNotsaleReason(OrderItemStock.NOTSALE_REASON_SUPPLIER_DOWN);
            }
        }


        return itemStock;
    }

}
