package order.service.operator.impl;

import goods.vo.SalesPriceCalVo;
import lombok.extern.slf4j.Slf4j;
import order.service.operator.OrgSalesPriceOperator;
import order.service.stubs.SalesPriceRateApiStub;
import order.vo.TempOrderItemVo;
import order.vo.TempOrderVo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import utils.Lang;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * @author Liang Wenxu
 * @since 2018/8/30
 */
@Slf4j
@Component("orgSalesPriceOperator")
public class DefaultOrgSalesPriceOperator implements OrgSalesPriceOperator {
    @Autowired
    SalesPriceRateApiStub salesPriceRateApi;

    /**
     * 使用费率计算之后的价格更新临时订单中的售价
     *
     * @param tempOrderVo 临时订单
     * @return
     */
    @Override
    public TempOrderVo updateTempOrder(TempOrderVo tempOrderVo) {
        long methodStartTime = System.currentTimeMillis();
        log.info("### 开始为订单计算第三方费率价格 ... ###");
        /** 使用费率表计算第三方价格 */
        if (!Lang.isEmpty(tempOrderVo.getOrganizationCode())) {
            long organizationTime = System.currentTimeMillis();
            String organizationId = tempOrderVo.getRealOrganizationId();
            log.info("查询机构信息耗时 {} ms", System.currentTimeMillis() - organizationTime);

            if (organizationId != null) {
                List<SalesPriceCalVo> salesPriceCalVoList = new ArrayList<>();

                // 按不同店铺分组
                for (TempOrderItemVo itemVo : tempOrderVo.getOrderItems()) {
                    salesPriceCalVoList.add(new SalesPriceCalVo(itemVo.getProductFashionId(), itemVo.getStoreId(),
                            itemVo.getStoreCode(), organizationId, itemVo.getSalePrice(),
                            itemVo.getCostPrice(), itemVo.getMarketPrice(),
                            itemVo.getCostPrice(), itemVo.getCount(), tempOrderVo.getOrganizationCode()));
                }


                long calBatchSalesPriceTime = System.currentTimeMillis();
                List<Map> rateCalRs = salesPriceRateApi.calBatchSalesPrice(salesPriceCalVoList);
                log.info("批量计算费率价格耗时 {} ms", System.currentTimeMillis() - calBatchSalesPriceTime);
                BigDecimal newOrderSalesPrice = new BigDecimal(0);
                BigDecimal newOrderCostPrice = new BigDecimal(0);
                BigDecimal newOrderMarketPrice = new BigDecimal(0);
                for (TempOrderItemVo itemVo : tempOrderVo.getOrderItems()) {
                    Optional<Map> searchRs = rateCalRs.stream().filter(
                            map -> itemVo.getProductFashionId().equals(map.get("fashionId"))).findFirst();
                    // 注意：如果销售金额被改动过的（营销活动）则不会修改salesPirce
                    if (searchRs.isPresent() && itemVo.getOrginSalePrice() != null && itemVo.getOrginSalePrice().equals(itemVo.getSalePrice())) {
                        Map m = searchRs.get();
                        itemVo.setSalePrice(((BigDecimal) m.get("salePrice")).setScale(2, BigDecimal.ROUND_UP));
                        itemVo.setSalesPriceRate((BigDecimal) m.get("rate"));
                        itemVo.setOrginSalePrice(itemVo.getSalePrice());
                        itemVo.recalateSumPrices();
                    }
                    newOrderSalesPrice = newOrderSalesPrice.add(itemVo.getSumSalePrice());
                    newOrderCostPrice = newOrderCostPrice.add(itemVo.getSumCostPrice());
                    newOrderMarketPrice = newOrderMarketPrice.add(itemVo.getSumMarketPrice());
                }

                tempOrderVo.setSumPrice(newOrderSalesPrice);
                tempOrderVo.setSumOrginalPrice(tempOrderVo.getSumPrice());
                tempOrderVo.setNeedPayPrice(tempOrderVo.getSumPrice().add(tempOrderVo.getSumShippingFee()));
                tempOrderVo.setSumCostPrice(newOrderCostPrice);
                tempOrderVo.setSumMarketPrice(newOrderMarketPrice);
            }
        }
        log.info("### 为订单计算第三方费率价格总耗时 {} ms", System.currentTimeMillis() - methodStartTime);
        return tempOrderVo;
    }
}
