package order.service.order.templates;

import activities.dto.goods.GoodsDto;
import com.alibaba.fastjson.JSON;
import com.google.common.base.Joiner;
import order.model.po.OrderItem;
import order.model.po.OrderMain;
import payment.model.po.Payment;
import goods.dto.product.ProductFashionDto;
import lombok.extern.slf4j.Slf4j;
import member.api.dto.core.CoreUserDto;
import member.api.dto.shop.MemberAddressDto;
import member.api.dto.shop.MemberDto;
import order.Contants;
import order.exceptions.IllegalAreaCodeException;
import order.exceptions.NoStockException;
import order.exceptions.OrderCreateException;
import order.exceptions.temporder.TempOrderCreateException;
import order.service.operator.OrderPriceOperator;
import order.service.operator.OrderStockOperator;
import order.service.order.helper.OrderHelper;
import order.service.order.helper.TempOrderHelper;
import order.service.service.OrderMainService;
import order.service.service.PaymentConfigService;
import order.service.service.PaymentService;
import order.service.service.query.OrderMainQueryService;
import order.service.stubs.*;
import order.vo.*;
import order.vo.response.OrderAuditingResult;
import order.vo.response.OrderPrice;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import outsideapi.exceptions.OrderCancelException;
import outsideapi.exceptions.SupplierInterfaceInvokeException;
import outsideapi.exceptions.SupplierOrderComfirmFalidException;
import outsideapi.vo.order.TrdOrder;
import outsideapi.vo.order.TrdOrderKey;
import outsideapi.vo.orderrequest.OrderRequst;
import sinomall.global.common.response.BaseResponse;
import sysmg.dto.SystemLogDto;
import utils.GlobalContants;
import utils.Lang;
import utils.data.BeanMapper;
import utils.web.UserAgent;
import utils.web.UserAgentUtil;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 商城默认订单处理模板
 * @author Liang Wenxu
 * @since 2018/7/10
 */
@Slf4j
public abstract class MallDefaultOrderTemplate extends OrderTemplateAdapter {
    // \b 是单词边界(连着的两个(字母字符 与 非字母字符) 之间的逻辑上的间隔),
    // 字符串在编译时会被转码一次,所以是 "\\b"
    // \B 是单词内部逻辑间隔(连着的两个字母字符之间的逻辑上的间隔)
    private static final String PHONE_REG = "\\b(ip(hone|od)|android|opera m(ob|in)i" + "|windows (phone|ce)|blackberry" + "|s(ymbian|eries60|amsung)|p(laybook|alm|rofile/midp" + "|laystation portable)|nokia|fennec|htc[-_]" + "|mobile|up.browser|[1-4][0-9]{2}x[1-4][0-9]{2})\\b";
    private static final String TABLE_REG = "\\b(ipad|tablet|(Nexus 7)|up.browser" + "|[1-4][0-9]{2}x[1-4][0-9]{2})\\b";

    // 移动设备正则匹配：手机端、平板
    private static Pattern phonePat = Pattern.compile(PHONE_REG, Pattern.CASE_INSENSITIVE);
    private static Pattern tablePat = Pattern.compile(TABLE_REG, Pattern.CASE_INSENSITIVE);

    // 依赖都放最后，太多了回影响阅读


    @Override
    public BaseResponse<Boolean> orderPreCheck(TempOrderVo tempOrderVo) {
        //TODO: 所有商城共用的校验，都放在这
        BaseResponse<Boolean> response = super.orderPreCheck(tempOrderVo);
        // 竞价订单校验
        List<String> skus = new ArrayList<>();
        tempOrderVo.getOrderItems().forEach(oi -> {
            skus.add(oi.getSku());
        });
        Boolean isExitsResult = biddingMainApi.findTempOrderBySkus(skus);

        // 亏本预校验：
        String realOrganizationCode = tempOrderVo.getOrganizationCode() == null ? "jicai" : tempOrderVo.getOrganizationCode();
        Vector<GoodsDto> goodsVector = new Vector<>();
        tempOrderVo.getOrderItems().stream()
                .filter(it -> !Lang.isEmpty(it.getGoodsId()))
                .forEach(it -> {
                    GoodsDto goodsDto = new GoodsDto();
                    goodsDto.setId(it.getGoodsId());
                    goodsVector.add(goodsDto);
                });
        List<Map<String, Object>> activityPriceBatch = activityGoodsApi.isActivityGoodsAndGetActivityPriceBatch(goodsVector, realOrganizationCode);
        // 价格判断以最新的价格为准
        try {
            OrderPrice orderPrice = orderPriceOperator.get(tempOrderVo);
            Vector<String> errorSkus = new Vector<>();
            orderPrice.getOrderItemPrices().forEach(oip -> {
                boolean isAct = false;
                Optional<TempOrderItemVo> itemOpt = tempOrderVo.getOrderItems().stream()
                        .filter( oi -> oi.getProductFashionId().equals(oip.getFashionId()))
                        .findFirst();

                if(itemOpt.isPresent()) {
                    if(activityPriceBatch != null) {
                        Optional<Map<String, Object>> activityPriceOpt = activityPriceBatch.stream()
                                .filter(m -> itemOpt.get().getGoodsId().equals(m.get("goodsId")))
                                .findFirst();
                        if(activityPriceOpt.isPresent()) {
                            isAct = (boolean) activityPriceOpt.get().get("isActivityGoods");
                            if (!isExitsResult && !isAct && oip.getSalesPrice().compareTo(oip.getCostPrice()) < 0) {
                                errorSkus.add(itemOpt.get().getSku());
                            }
                        }
                    }
                }
            });
            if(errorSkus.size() > 0) {
                //  校验失败
                response.setSuccess(true);
                response.setResultCode(Contants.ORDER_PRECHECK_RESP_CODES.SALES_PRICE_TOO_LOW.getValue());
                response.setResultMessage(Contants.ORDER_PRECHECK_RESP_CODES.SALES_PRICE_TOO_LOW.getMessage()
                        .replace("[]", "["+Joiner.on(",").join(errorSkus)+"]"));
                response.setResult(false);
            }
        } catch (IllegalAreaCodeException e) {
            response.setSuccess(false);
            response.setResultCode(Contants.ORDER_PRECHECK_RESP_CODES.SALES_PRICE_QUERY_ERROR.getValue());
            response.setResultMessage(Contants.ORDER_PRECHECK_RESP_CODES.SALES_PRICE_QUERY_ERROR.getMessage()+ ", " + e.getMessage());
            response.setResult(false);
            e.printStackTrace();
        }
        return super.orderPreCheck(tempOrderVo);
    }

    /**
     * 创建临时订单，统一默认实现，在其子类覆盖此方法以实现个性化
     * <p>将各子类都要用到的公共代码抽象到此处，各业务子类只需覆盖此方法并在进行扩展即可</p>
     * @param tempOrderCreateRequest
     * @return
     */
    @Override
    public TempOrderVo createTempOrder(final TempOrderCreateRequest tempOrderCreateRequest) throws TempOrderCreateException {
        tempOrderCreateRequest.getGoodsInfoVos();
        TempOrderVo tempOrderVo = BeanMapper.map(tempOrderCreateRequest, TempOrderVo.class);
        tempOrderVo.setTmpOrderNo(tempOrderHelper.genTempOrderNo());
        tempOrderVo.setOrderItems(new ArrayList<>());

        String[] areaCodeArray = new String[]{null, null, null, null};
        // 地区编码
        if(!Lang.isEmpty(tempOrderCreateRequest.getTownCode())) {
            tempOrderVo.setAreaCode(tempOrderCreateRequest.getTownCode());
            areaCodeArray[3] = tempOrderCreateRequest.getTownCode();
        }
        if(!Lang.isEmpty(tempOrderCreateRequest.getCountyCode())) {
            if(Lang.isEmpty(tempOrderVo.getAreaCode())) {
                tempOrderVo.setAreaCode(tempOrderCreateRequest.getCountyCode());
            }
            areaCodeArray[2] = tempOrderCreateRequest.getCountyCode();
        }
        if(!Lang.isEmpty(tempOrderCreateRequest.getCityCode())) {
            if(Lang.isEmpty(tempOrderVo.getAreaCode())) {
                tempOrderVo.setAreaCode(tempOrderCreateRequest.getCityCode());
            }
            areaCodeArray[1] = tempOrderCreateRequest.getCityCode();
        }
        if(!Lang.isEmpty(tempOrderCreateRequest.getProvinceCode())) {
            if(Lang.isEmpty(tempOrderVo.getAreaCode())) {
                tempOrderVo.setAreaCode(tempOrderCreateRequest.getProvinceCode());
            }
            areaCodeArray[0] = tempOrderCreateRequest.getProvinceCode();
        }
        tempOrderVo.setAreaCodeList(Arrays.asList(areaCodeArray));
        // 处理商品列表

        // 获取商品信息
        List<ProductFashionDto> productFashionDtos = productFashionApi.findByIds(
                tempOrderCreateRequest.getGoodsInfoVos().stream().map(GoodsInfoVo::getProductFashionId).collect(Collectors.toList()));
        Map<String, ProductFashionDto> productFashionDtoMap = productFashionDtos.stream().collect(Collectors.toMap(ProductFashionDto::getId, Function.identity()));
        for(GoodsInfoVo goodsInfoVo : tempOrderCreateRequest.getGoodsInfoVos()) {
            ProductFashionDto productFashion = productFashionDtoMap.get(goodsInfoVo.getProductFashionId());
            TempOrderItemVo tempOrderItemVo = createTempOrderItem(goodsInfoVo, productFashion);
            tempOrderVo.getOrderItems().add(tempOrderItemVo);
        }

        // 更新订单金额

        // 计算订单金额
        OrderSumItemPrices orderSumItemPrices = tempOrderHelper.calculateSumItemPrices(tempOrderVo.getOrderItems());

        tempOrderVo.setSumPrice(orderSumItemPrices.getSumPrice());
        tempOrderVo.setSumOrginalPrice(orderSumItemPrices.getSumOrginalPrice());
        tempOrderVo.setSumCostPrice(orderSumItemPrices.getSumCostPrice());
        tempOrderVo.setSumMarketPrice(orderSumItemPrices.getSumMarketPrice());
        tempOrderVo.setSumShippingFee(new BigDecimal(0));
        tempOrderVo.setOrganizationCode(tempOrderCreateRequest.getOrganizationCode());
        tempOrderVo.setRealOrganizationId(tempOrderCreateRequest.getOrgainzationId());

        // 处理浏览器数据
        UserAgent agent = UserAgentUtil.getUserAgent(tempOrderCreateRequest.getUserAgent());
        tempOrderVo.setBrowserType(agent.getBrowserType());
        tempOrderVo.setPlatformType(agent.getPlatformType());
        // 匹配
        Matcher matcherPhone = phonePat.matcher(tempOrderCreateRequest.getUserAgent());
        Matcher matcherTable = tablePat.matcher(tempOrderCreateRequest.getUserAgent());
        if (matcherTable.find()) {
            tempOrderVo.setBuyType(TempOrderVo.IPAD);
        } else if (matcherPhone.find()) {
            tempOrderVo.setBuyType(TempOrderVo.MOBILE);
        } else {
            tempOrderVo.setBuyType(TempOrderVo.PC);
        }

        // 生成主订单后，根据店铺拆分订单
        List<TempSubOrderVo> splitedOrder = splitOrder(tempOrderVo.getOrderItems(), tempOrderVo.getTmpOrderNo());

        splitedOrder.forEach(o -> {
            o.setOrgCode(tempOrderCreateRequest.getOrganizationCode());
            o.setGoodsType(tempOrderCreateRequest.getGoodsType());
            o.setOrderMode(tempOrderCreateRequest.getOrderMode());
        });
        tempOrderVo.setSubOrders(splitedOrder);

        return tempOrderVo;
    }

    /**
     * 拆分主订单
     * @param allItems 主订单商品
     * @return 拆分后的子订单列表
     */
    protected List<TempSubOrderVo> splitOrder(List<TempOrderItemVo> allItems, String parentOrderNo) {
        // 先分个组
        Map<String, List<TempOrderItemVo>> orderItemGroup = allItems.stream()
                .collect(Collectors.groupingBy(TempOrderItemVo::getStoreId));
        // jdk8以下，可以用guava替代
//        Map<String, List<TempOrderItemVo>> group = Multimaps.asMap(
//                Multimaps.index(allItems, new com.google.common.base.Function<TempOrderItemVo, String>() {
//                    @Override
//                    public String apply(TempOrderItemVo tempOrderItemVo) {
//                        return tempOrderItemVo.getStoreId();
//                    }
//                })
//        );

        // 默认的拆分规则为按店铺拆分
        List<TempSubOrderVo> subOrderList = new ArrayList<>();
        orderItemGroup.forEach((s, tempOrderItemVos) -> {
            if(tempOrderItemVos.size() > 0 && tempOrderItemVos.get(0) != null) {
                TempSubOrderVo tempSubOrderVo = new TempSubOrderVo();
                tempSubOrderVo.setPOrderNo(parentOrderNo);
                tempSubOrderVo.setTmpOrderNo(tempOrderHelper.genTempOrderNo());
                tempSubOrderVo.setOrderItems(tempOrderItemVos);
                tempSubOrderVo.setStoreId(s);
                tempSubOrderVo.setStoreCode(tempOrderItemVos.get(0).getStoreCode());
                // 计算子订单金额
                OrderSumItemPrices orderSumItemPrices = tempOrderHelper.calculateSumItemPrices(tempOrderItemVos);
                tempSubOrderVo.setSumPrice(orderSumItemPrices.getSumPrice());
                tempSubOrderVo.setSumOrginalPrice(orderSumItemPrices.getSumOrginalPrice());
                tempSubOrderVo.setSumCostPrice(orderSumItemPrices.getSumCostPrice());
                tempSubOrderVo.setSumMarketPrice(orderSumItemPrices.getSumMarketPrice());

                tempSubOrderVo.setCreateTime(new Date());

                subOrderList.add(tempSubOrderVo);
            }
        });
//        for(Map.Entry<String, List<TempOrderItemVo>> entry : orderItemGroup.entrySet()) {
//
//        }



        return subOrderList;
    }


    /**
     * 创建临时订单商品实例
     * @param goodsInfoVo
     * @param fashion
     * @return
     */
    protected TempOrderItemVo createTempOrderItem(GoodsInfoVo goodsInfoVo, ProductFashionDto fashion) {
        TempOrderItemVo tempOrderItemVo = new TempOrderItemVo();
        tempOrderItemVo.setProductFashionId(fashion.getId());
        tempOrderItemVo.setProductId(fashion.getProduct().getId());
        tempOrderItemVo.setStoreId(fashion.getProduct().getStoreId());
        tempOrderItemVo.setSalePrice(fashion.getSalePrice());
        tempOrderItemVo.setMarketPrice(fashion.getMarketPrice());
        tempOrderItemVo.setCostPrice(fashion.getCostPrice());
        tempOrderItemVo.setFashionPic(fashion.getFashionPic());
        tempOrderItemVo.setCount(goodsInfoVo.getCount());
        if (!Lang.isEmpty(fashion.getGoods())) {
            tempOrderItemVo.setFashionTitle(fashion.getGoods().getName());
        } else {
            tempOrderItemVo.setFashionTitle(Lang.isEmpty(fashion.getFashionTitle()) ? fashion.getProduct().getProductName() : fashion.getFashionTitle());
        }
        tempOrderItemVo.setFieldCode(fashion.getFieldCode());
        tempOrderItemVo.setFieldName(fashion.getFieldName());
        tempOrderItemVo.setFieldType(fashion.getFieldType());
        tempOrderItemVo.setValueName(fashion.getValueName());
        tempOrderItemVo.setGoodsId(goodsInfoVo.getGoodsId());
        tempOrderItemVo.setValue(fashion.getValue());
        tempOrderItemVo.setLackRemind(fashion.getLackRemind());
        tempOrderItemVo.setStockCount(new Long(fashion.getStockCount()));
        tempOrderItemVo.setProductCode(fashion.getProductCode());
        tempOrderItemVo.setSku(fashion.getProductCode());
        tempOrderItemVo.setStoreId(goodsInfoVo.getStoreId());
        tempOrderItemVo.setStoreCode(goodsInfoVo.getStoreCode());
        tempOrderItemVo.setStoreType(goodsInfoVo.getStoreType());

        // 处理配件
        if(goodsInfoVo.getChildren() != null && !goodsInfoVo.getChildren().isEmpty()) {
            tempOrderItemVo.setChildren(new ArrayList<>());

            for(GoodsInfoVo gi : goodsInfoVo.getChildren()) {
                Optional<ProductFashionDto> optional = fashion.getChildren().stream()
                        .filter(it -> it.getId() == gi.getProductFashionId()).findFirst();
                if(optional.isPresent()) {
                    tempOrderItemVo.getChildren().add(createTempOrderItem(gi, optional.get()));
                }
            }

            // 计算组合价格
            BigDecimal combinationSalePrice = new BigDecimal(0);
            BigDecimal combinationMarketPrice = new BigDecimal(0);
            BigDecimal combinationCostPrice = new BigDecimal(0);
            for(TempOrderItemVo subItems : tempOrderItemVo.getChildren()) {
                // 每个组合的单价为组合下所有子商品的总价
                combinationSalePrice = combinationSalePrice.add(subItems.getSumSalePrice());
                combinationMarketPrice = combinationMarketPrice.add(subItems.getSumMarketPrice());
                combinationCostPrice = combinationCostPrice.add(subItems.getSumCostPrice());
            }

            tempOrderItemVo.setSalePrice(combinationSalePrice);
            tempOrderItemVo.setMarketPrice(combinationMarketPrice);
            tempOrderItemVo.setCostPrice(combinationCostPrice);
        }
        tempOrderItemVo.recalateSumPrices();
        tempOrderItemVo.setOrginSalePrice(tempOrderItemVo.getSalePrice());
        return tempOrderItemVo;
    }

    /**
     * 创建订单，统一默认实现，在其子类覆盖此方法以实现个性化
     * <p><b>Notic:</b>注意，在进入这一步前，请确保request中的所有金额、运费都已经计算到位！</p>
     * <p><b>Notic:</b>注意，创建订单会同时创建Payment信息，在子类覆盖此方法的时候，对于金额的变更请慎重处理！</p>
     * @param request
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public OrderVo createOrder(CreateOrderRequest request) throws OrderCreateException, NoStockException {
        long methodStartTime = System.currentTimeMillis();
        log.info("### 开始创建正式订单 ... ###");
        log.info("创建订单 temp 数据 tempOrderVo -> {}",JSON.toJSONString(request));

        OrderVo orderVo = null;

        Long memberTime = System.currentTimeMillis();
        String coreUserId = request.getUserId();
        String coreUserOrganizationId = memberApi.findOrganizationIdById(request.getMemberId());
        CoreUserDto coreUser = new CoreUserDto();
        coreUser.setId(coreUserId);
        coreUser.setOrganizationId(coreUserOrganizationId);
        MemberDto member = new MemberDto();
        member.setId(request.getMemberId());
        member.setCoreUser(coreUser);
        log.info("获取 member 耗时 {} ms", System.currentTimeMillis() - memberTime);

        // 参数校验
        if(request.getSubOrder() == null) {
            throw new OrderCreateException(Contants.ORDER_CREATE_ERROR.SUBORDER_NOT_PROVIDED);
        }
        if(request.getSubOrder().getOrderItems() == null || request.getSubOrder().getOrderItems().size() <= 0) {
            throw new OrderCreateException(Contants.ORDER_CREATE_ERROR.GOODS_NOT_PROVIDED);
        }
        Payment payment;

        String realOrganizationId = request.getRealOrganizationId();
        String realOrganizationCode = null;
        if (!Lang.isEmpty(realOrganizationId)) {
            long organizationTime = System.currentTimeMillis();
            realOrganizationCode = organizationApi.findOrganizationCodeById(request.getRealOrganizationId());
            log.info("获取机构信息耗时 {} ms", System.currentTimeMillis() - organizationTime);
        }

        List<TempSubOrderVo> subOrders = new ArrayList<>();
        if(request.getSubOrder().getSubOrders() != null && request.getSubOrder().getSubOrders().size() > 0) {
            subOrders.addAll(request.getSubOrder().getSubOrders());
        } else {
            subOrders.add(request.getSubOrder());
        }

        List<OrderVo> sucOrders = new ArrayList<>();
        MemberAddressDto memberAddress = MemberAddressApi.findById(request.getMemberAddrId());
        for(TempSubOrderVo cOrder : subOrders) {
            long genOrderTime = System.currentTimeMillis();
            OrderMain order = orderHelper.genOrder(request, cOrder, member);

            //生成订单信息
            log.info("生成订单信息耗时 {} ms", System.currentTimeMillis() - genOrderTime);

//            /**Akers 2017-12-01修改： 校验价格是否正常 START */
            boolean hasPriceError = false;
            List<String> goodsIds = new ArrayList<>();
            for (OrderItem item : order.getOrderItems()) {
                //协议商品不存在于goods表
                if (!Lang.isEmpty(item.getGoodsId())) {
                    goodsIds.add(item.getGoodsId());
                }
            }

            // 生成支付信息
            payment = new Payment();
            payment.setConfig(paymentConfigService.findByCode(request.getPaymentCode()));
            payment.setStoreId(cOrder.getStoreId());
            payment.setUserId(request.getUserId());
            payment.setType("1");
            payment.setMoney(cOrder.getSumPrice());
            if (!Lang.isEmpty(order.getPayMethod()) && order.getPayMethod().equals(OrderMain.PAY_METHOD_SCORE)) {
                payment.setStatus(Payment.PAY_STATUS_PAID);
            } else {
                payment.setStatus(Payment.PAY_STATUS_WAIT_TO_PAY);
            }
            payment.setOrderId(order.getId());
            payment = paymentService.saveOrUpdate(payment, true);
            order.setPaymentId(payment.getId());
            // 持久化order
            OrderVo orderVo1 = orderMainService.toOrderVo(orderMainService.saveOrder(order, true), payment, memberAddress);
            orderVo1.setOrganizationCode(request.getOrganizationCode());
            sucOrders.add(orderVo1);
        }

        //判断是否需要父订单
        if(request.getSubOrder().getSubOrders() != null && !Lang.isEmpty(request.getSubOrder().getSubOrders())) {
            OrderVo pOrder = BeanMapper.map(sucOrders.get(0), OrderVo.class);
            // 填充子单的item到父订单
            pOrder.setOrderItems(sucOrders.stream()
                    .flatMap(o -> o.getOrderItems().stream())
                    .collect(Collectors.toList()));
            pOrder.setSubOrders(sucOrders);
            pOrder.setOrderId(null);
            pOrder.setOrderNo(orderHelper.generateOrderNo());
            // 合并运费
            pOrder.setFreight(sucOrders.stream().map(OrderVo::getFreight).reduce(BigDecimal.ZERO, BigDecimal::add));
            // 合并无运费订单总价
            pOrder.setSumNofreightPrice(sucOrders.stream().map(OrderVo::getSumNofreightPrice).reduce(BigDecimal.ZERO, BigDecimal::add));
            pOrder.setSumPrice(pOrder.getSumNofreightPrice().add(pOrder.getFreight()));

        } else {
            orderVo = sucOrders.get(0);
        }

        //TODO: 删除购物车商品改为在Web调用（OrderApi和CartApi的业务隔离）

        //TODO: 保存父订单
        // 保存ParentOrder
        // 关联生成的所有OrderMain


        log.info("### 创建正式订单总耗时 {} ms ###", System.currentTimeMillis() - methodStartTime);

        return orderVo;
    }

    /**
     * 订单支付成功
     * @param orderVo
     * @return
     */
    @Override
    public OrderPayConfirmResp paySubmit(OrderVo orderVo) throws SupplierInterfaceInvokeException, SupplierOrderComfirmFalidException {
        OrderRequst orderRequst = BeanMapper.map(orderVo, BeanMapper.getType(OrderVo.class), BeanMapper.getType(OrderRequst.class));
        OrderPayConfirmResp response = new OrderPayConfirmResp();
        supplierOrderApi.confirmOrder(orderRequst);
        orderMainService.updateBatchStatusByOrderNo(new ArrayList<String>(){{add(orderVo.getOrderNo());}}, GlobalContants.ORDER_STATUS.OUNFILLED);
        response.setSuccess(true);
        return response;
    }

    @Override
    public BaseResponse<CancelOrderResp> cancelOrder(List<OrderKey> orderKeys, Contants.CancelOrderType cancelType) {
        List<String> orderKeyStrs = orderKeys.stream()
                .map(orderKey ->
                        Joiner.on("_").skipNulls()
                                .join(orderKey.getOrderId()
                                        , orderKey.getOrderNo())).collect(Collectors.toList());
        /**Akers Liang 2017-05-02修改： 订单被取消时保存系统日志 START */
        ((Runnable) () -> {
            SystemLogDto systemLog = new SystemLogDto();
            systemLog.setLogType(SystemLogDto.LOG_TYPE_INFO);
            systemLog.setPostUrl("CANCEL_ORDER" + Joiner.on(",").join(orderKeyStrs));
            systemLog.setLogSource(SystemLogDto.LOG_SOURCE_JICAI);
            systemLog.setErrorDesc(cancelType.toString());
            systemLog.setPostUrl("orderNo: " + Joiner.on(",").join(orderKeyStrs) + ", cancelType: " + cancelType.toString());

            systemLogApi.saveBean(systemLog);
        }).run();
        /**Akers Liang 2017-05-02修改： 订单被取消时保存系统日志 END */
        long sucCount = 0;
        long failCount = 0;
        BaseResponse<CancelOrderResp> resp = new BaseResponse<>();
        CancelOrderResp cancelOrderResp = new CancelOrderResp();
        List<CancelOrderDetailResp> detailResps = new ArrayList<>();
        if(orderKeys != null && orderKeys.size() > 0) {
            List<OrderMain> orderMainList = orderMainQueryService.findByKey(orderKeys);
            for(OrderMain om : orderMainList) {
                CancelOrderDetailResp detail = new CancelOrderDetailResp();
                detail.setOrderNo(om.getOrderNo());
                try {
                    // 第三方取消  如果没有第三方订单号，不应该去供应商取消

                    if(!Lang.isEmpty(om.getThirdOrderNo())){
                        TrdOrderKey trdOrderKey = new TrdOrderKey();
                        trdOrderKey.setTrdOrderNo(om.getThirdOrderNo());
                        trdOrderKey.setStoreCode(om.getStoreCode());
                        supplierOrderApi.cancelOrder(trdOrderKey);
                    }

                    // 商城取消
                    int effected = orderMainService.cancelOrder(new OrderKey(om.getId()), cancelType);
                    if(effected > 0) {
                        cancelOrderResp = new CancelOrderResp();
                        sucCount += 1;
                        detail.setSuccessed(true);
                    } else {
                        failCount += 1;
                        detail.setSuccessed(false);
                        detail.setErrorCode(Contants.ORDER_CANCEL_ERROR.ORDER_NOFOUND.getValue());
                        detail.setMessage(Contants.ORDER_CANCEL_ERROR.ORDER_NOFOUND.getMessage());
                    }
                } catch (OrderCancelException oce) {
                    detail.setMessage(oce.getMessage());
                    detail.setSuccessed(false);
                    detail.setErrorCode(oce.getErrorCode().getValue());
                } catch (Exception e) {
                    failCount += 1;
                    detail.setMessage(e.getMessage());
                    detail.setSuccessed(false);
                    detail.setErrorCode(Contants.ORDER_CANCEL_ERROR.ORDER_SQL_EXCEPTION.getValue());
                }
                detailResps.add(detail);
            }
        }
        cancelOrderResp.setSuccessCounts(sucCount);
        cancelOrderResp.setFailureCounts(failCount);
        cancelOrderResp.setDetails(detailResps);
        resp.setSuccess(true);
        resp.setResult(cancelOrderResp);
        return resp;
    }

    /**
     * 订单审核（批量），审核通过订单可进入生产状态（待支付），审核不通过则取消订单
     * <p><b>Notic:各子类需自行处理状态变更，切记切记</b></p>
     * @param orderKeys           需操作的订单索引
     * @param passed              true|false（审核通过|审核不通过）
     * @return 返回中包括各订单的审核处理情况
     */
    @Override
    public List<OrderAuditingResult> audit(List<OrderKey> orderKeys, Boolean passed) {
        List<OrderAuditingResult> r = new ArrayList<>();
        List<OrderMain> orderMains = orderMainQueryService.findByKey(orderKeys);
        for(OrderMain m : orderMains) {
            if(m.getStatus().equals(GlobalContants.ORDER_STATUS.CANCEL.getValue())
                    || m.getStatus().equals(GlobalContants.ORDER_STATUS.OUTOFTIME_CANCEL.getValue())
                    || m.getStatus().equals(GlobalContants.ORDER_STATUS.ERROR.getValue())) {
                OrderAuditingResult oar = new OrderAuditingResult();
                oar.setErrorCode(Contants.ORDER_AUDITING_ERROR.NOTAUDITABLE.getValue());
                oar.setMessage("订单已取消");
                oar.setOrderKey(new OrderKey(m.getId(),  m.getOrderNo()));
                oar.setSuccess(false);
                r.add(oar);
                orderKeys.removeIf(i -> m.getId().equals(i.getOrderId()) || m.getOrderNo().equals(i.getOrderNo()));
            } else if (!m.getStatus().equals(GlobalContants.ORDER_STATUS.NEWORDER.getValue())) {
                OrderAuditingResult oar = new OrderAuditingResult();
                oar.setMessage("非可审核订单");
                oar.setErrorCode(Contants.ORDER_AUDITING_ERROR.NOTAUDITABLE.getValue());
                oar.setOrderKey(new OrderKey(m.getId(),  m.getOrderNo()));
                oar.setSuccess(false);
                r.add(oar);
                orderKeys.removeIf(i -> m.getId().equals(i.getOrderId()) || m.getOrderNo().equals(i.getOrderNo()));
            } else {
                // 符合审核条件
                OrderAuditingResult oar = new OrderAuditingResult();
                oar.setOrderKey(new OrderKey(m.getId(),  m.getOrderNo()));
                oar.setSuccess(true);
                r.add(oar);
            }
        }

        if(passed) {
            List<OrderKey> sucOrders = new ArrayList<>();
            r.stream().filter(ri -> ri.getSuccess()).forEach(suc -> {
                sucOrders.add(suc.getOrderKey());
            });
            // 通过审核
            orderMainService.updateBatchStatusByOrderNo(
                    sucOrders.stream().map(OrderKey::getOrderNo).collect(Collectors.toList())
                    , GlobalContants.ORDER_STATUS.OBLIGATION);
        } else {
            // 审核不通过
            orderMainService.updateBatchStatusByOrderNo(
                    orderKeys.stream().map(OrderKey::getOrderNo).collect(Collectors.toList())
                    , GlobalContants.ORDER_STATUS.NO_PASS);
        }
        return r;
    }

    /**
     * 刷新订单状态
     *
     * @param orderKeys
     * @return
     */
    @Override
    public BaseResponse<List<OrderVo>> refresh(List<OrderKey> orderKeys) {
        BaseResponse<List<OrderVo>> baseResponse = new BaseResponse<>();
        List<OrderVo> result = new ArrayList<>();
        baseResponse.setSuccess(true);
        List<OrderMain> orderMainList = orderMainQueryService.findByKey(orderKeys);

        for(OrderMain m : orderMainList) {
            TrdOrderKey trdOrderKey = new TrdOrderKey();
            trdOrderKey.setStoreCode(m.getStoreCode());
            trdOrderKey.setTrdOrderNo(m.getThirdOrderNo());
            try {
                TrdOrder trdOrder = supplierOrderApi.getDetail(trdOrderKey);
                orderMainService.updateTrdOrderInfo(new OrderKey(m.getId(), m.getOrderNo()), trdOrder);
            } catch (SupplierInterfaceInvokeException e) {
                // 供应商接口调用出错
                baseResponse.setSuccess(false);
                baseResponse.setResultCode(e.getErrorCode().getValue());
                baseResponse.setResultMessage(e.getMessage());
                log.error("供应商接口调用报错", e);
            }
        }



        return super.refresh(orderKeys);
    }

    @Autowired
    SupplierOrderApiStub supplierOrderApi;

    @Autowired
    OutsideProductApiStub outsideProductApi;

    @Autowired
    TempOrderHelper tempOrderHelper;

    // 注入API时注入本地代理
    @Autowired
    ProductFashionApiStub productFashionApi;

    @Autowired
    OrderStockOperator orderStockOperator;

    @Autowired
    MemberApiStub memberApi;

    @Autowired
    OrganizationApiStub organizationApi;

    @Autowired
    CoreCompDepartUserApiStub coreCompDepartUserApi;

    @Autowired
    CoreCompanyApiStub coreCompanyApi;

    @Autowired
    OrderHelper orderHelper;

    @Autowired
    PaymentConfigService paymentConfigService;

    @Autowired
    PaymentService paymentService;

    @Autowired
    MemberAddressApiStub MemberAddressApi;

    @Autowired
    SystemLogApiStub systemLogApi;

    @Autowired
    OrderMainService orderMainService;

    @Autowired
    OrderMainQueryService orderMainQueryService;

    @Autowired
    BiddingMainApiStub biddingMainApi;

    @Autowired
    ActivityGoodsApiStub activityGoodsApi;

    @Autowired
    OrderPriceOperator orderPriceOperator;
}
