package order.service.service;

import order.Contants;
import order.exceptions.IllegalAreaCodeException;
import order.exceptions.OrderCreateException;
import order.exceptions.temporder.TempOrderCreateException;
import order.vo.*;
import order.vo.response.OrderAuditingResult;
import order.vo.response.OrderPrice;
import order.vo.response.OrderStock;

import java.util.List;

/**
 * OrderService订单业务服务接口，定义于OrderApi对应的订单相关业务
 * <p><b>此Service只包括订单的相关业务，按业务场景定义方法</b></p>
 * <p>订单通用流程：1.临时订单->2.提交订单->3.订单审核（包括集采订单的工作流审核，无审核要求的商城，直接审核通过即可）->4.待付款->5.付款成功</p>
 * <p>直接数据操作可使用OrderMainService，数据查询使用OrderMainQueryService</p>
 *
 * @author Liang Wenxu
 * @since 2018/7/12
 */
public interface OrderService {
    /**
     * 创建临时订单
     * <p>此接口传入所需购买的所有商品（批量下单与普通下单统一入口）</p>
     *
     * @param tempOrderCreateRequest 只传入客户勾选后的待购商品
     * @return 临时订单实例，如是批量下单、提交的商品中包含不同店铺等需生成多个订单的场景则临时订单会有父子订单，注意与商城展示用的临时订单Vo区分噢
     */
    TempOrderVo newTempOrder(final TempOrderCreateRequest tempOrderCreateRequest) throws TempOrderCreateException;

    /**
     * 临时订单库存校验
     *
     * @param tempOrderVo 临时订单对象
     */
    OrderStock tempOrderStock(TempOrderVo tempOrderVo) throws IllegalAreaCodeException;

    /**
     * 临时定案价格重算
     *
     * @param tempOrderVo
     * @return
     */
    OrderPrice tempOrderPrice(TempOrderVo tempOrderVo) throws IllegalAreaCodeException;

    /**
     * 提交订单，将临时订单提交为真实订单并预占商品库存（如需要）
     * <p>在结算确认页面，用户确认操作后调用，将临时订单确认为真实订单</p>
     *
     * @param tempOrder 临时订单
     * @return 返回OrderVo中将包括订单的简略信息，如需详情请调用OrderQueryApi中相关API进行查询
     */
    OrderVo submit(TempOrderVo tempOrder) throws OrderCreateException;

    /**
     * 刷新订单状态
     * <p>支持批量，请注意控制每批数量，供应商接口是单个订单处理的，不推荐一次处理超过100订单</p>
     * @param orderKeys，订单索引主键
     * @return
     */
    List<OrderVo> refresh(List<OrderKey> orderKeys);

    /**
     * 订单审核（单个）
     * <p>订单审核通过后，即可进入支付环节</p>
     *
     * @param orderKey            需操作的订单索引
     * @param passed              true|false（审核通过|审核不通过）
     * @param cancelSupplierOrder 审核不通过时是否取消供应商订单 true|false（取消|不取消）
     * @return 订单的处理结果
     */
    OrderAuditingResult audit(OrderKey orderKey, Boolean passed, Boolean cancelSupplierOrder);

    /**
     * 订单审核（支持批量）
     *
     * @param orderKeys           需操作的订单索引
     * @param passed              true|false（审核通过|审核不通过）
     * @param cancelSupplierOrder 审核不通过时是否取消供应商订单 true|false（取消|不取消）
     * @return 返回各订单的处理结果
     */
    List<OrderAuditingResult> audit(List<OrderKey> orderKeys, Boolean passed, Boolean cancelSupplierOrder);

    /**
     * 订单支付预处理完毕
     * （客户已开始支付，但未支付成功时调用）
     *
     * @param orderKey   订单key，orderNo或orderId，优先id
     * @param cusPayInfo 客户支付信息，如需在客户支付环节带入新的信息从此参数带入
     */
    void prePayDone(OrderKey orderKey, CusPayInfo cusPayInfo);

    /**
     * 订单支付确认，确认订单已收款并进入发货流程
     *
     * @param orderKey 其中orderNo、orderId可任选其一
     * @return
     */
    OrderPayConfirmResp payComfirmed(OrderKey orderKey);

    /**
     * 取消指定的订单
     * <p><b>Notic:</b>此方法根据{@link OrderTemplateConfigKey}进行适配</p>
     *
     * @param orderVos   其中OrderKey只需提供orderNo或orderId
     * @param cancelType
     * @return
     */
    CancelOrderResp cancel(List<OrderKey> orderVos, Contants.CancelOrderType cancelType);

    /**
     * 订单收货
     *
     * @param orderKey    其中orderNo、orderId可任选其一
     * @param receiptType 订单收货类型，见{@link Contants}中ORDER_RECEIPT_TYPE_开头常量
     * @return
     */
    void receipt(OrderKey orderKey, String receiptType);

    /**
     * 订单状态改变，系统级改变，改变订单状态并会产生OrderStatusChangeEvent，其operator固定为{@link Contants#ORDER_OPERATOR_SYSTEM}
     *
     * @param orderStatus 订单状态
     */
    void statusChange(OrderStatus orderStatus);

}