package order.service.service.impl;

import com.google.common.base.Joiner;
import order.model.po.OrderMain;
import payment.model.po.Payment;
import lombok.extern.slf4j.Slf4j;
import member.api.dto.shop.MemberAddressDto;
import order.Contants;
import order.exceptions.IllegalAreaCodeException;
import order.exceptions.NoStockException;
import order.exceptions.OrderCreateException;
import order.exceptions.temporder.TempOrderCreateException;
import order.model.repository.OrderRepos;
import order.service.events.OrderCreateEvent;
import order.service.events.OrderShippingComfirmedEvent;
import order.service.events.OrderStatusChangeEvent;
import order.service.factory.OrderTemplateFactory;
import order.service.operator.OrderFreightOperator;
import order.service.operator.OrderPriceOperator;
import order.service.operator.OrderStockOperator;
import order.service.operator.OrgSalesPriceOperator;
import order.service.order.helper.OrderHelper;
import order.service.order.helper.TempOrderHelper;
import order.service.order.templates.OrderTemplate;
import order.service.service.OrderMainService;
import order.service.service.OrderService;
import order.service.service.PaymentService;
import order.service.service.query.OrderMainQueryService;
import order.service.stubs.MemberAddressApiStub;
import order.service.stubs.OrganizationApiStub;
import order.service.stubs.ProductFashionApiStub;
import order.service.utils.OrderTemplateFactoryUtil;
import order.vo.*;
import order.vo.response.OrderAuditingResult;
import order.vo.response.OrderFreight;
import order.vo.response.OrderPrice;
import order.vo.response.OrderStock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import outsideapi.exceptions.SupplierInterfaceInvokeException;
import outsideapi.exceptions.SupplierOrderComfirmFalidException;
import sinomall.global.common.response.BaseResponse;
import utils.GlobalContants;
import utils.Lang;
import utils.data.BeanMapper;
import utils.data.Beans;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * 一般的、正常的、OrderService实现
 *
 * @author Liang Wenxu
 * @since 2018/7/12
 */
@Component("orderService")
@Slf4j
@Transactional(readOnly = true, rollbackFor = Exception.class)
public class DefaultOrderServiceImpl implements OrderService {

    @Autowired
    OrderTemplateFactoryUtil orderTemplateFactoryUtil;

    @Autowired
    TempOrderHelper tempOrderHelper;

    @Autowired
    OrgSalesPriceOperator orgSalesPriceOperator;

    @Autowired
    ProductFashionApiStub productFashionApi;

    @Autowired
    OrderStockOperator orderStockOperator;

    @Autowired
    OrderPriceOperator orderPriceOperator;

    @Autowired
    OrderFreightOperator orderFreightOperator;

    @Autowired
    OrderRepos orderRepos;

    @Autowired
    OrderHelper orderHelper;

    @Autowired
    OrganizationApiStub organizationApi;

    @Autowired
    OrderMainQueryService orderMainQueryService;

    @Autowired
    PaymentService paymentService;

    @Autowired
    MemberAddressApiStub memberAddressApi;

    @Autowired
    OrderMainService orderMainService;

    @Override
    public TempOrderVo newTempOrder(final TempOrderCreateRequest tempOrderCreateRequest) throws TempOrderCreateException {
        log.info("orderService.newTempOrder");
        TempOrderVo createdTempOrder = null;

        if (tempOrderCreateRequest.getGoodsInfoVos() != null) {
            tempOrderCreateRequest.setGoodsInfoVos(tempOrderCreateRequest.getGoodsInfoVos().stream().filter(Objects::nonNull).collect(Collectors.toList()));
            if (tempOrderCreateRequest.getGoodsInfoVos().isEmpty()) {
                throw new TempOrderCreateException(Contants.TEMPORDER_CREATE_ERROR.GOODS_NOT_PROVIDED);
            }
        } else {
            throw new TempOrderCreateException(Contants.TEMPORDER_CREATE_ERROR.GOODS_NOT_PROVIDED);
        }

        // 按GOODS_TYPE提取商品
        Map<String, List<GoodsInfoVo>> groupingGoods = tempOrderCreateRequest.getGoodsInfoVos().stream().collect(Collectors.groupingBy(GoodsInfoVo::getGoodsType));
        // 判断是否包含不同性质商品
        boolean hasMutileGoodsTypes = false;
        if (groupingGoods.entrySet().size() > 1) {
            hasMutileGoodsTypes = true;
        }

        OrderTemplateConfigKey orderTemplateConfigKey = new OrderTemplateConfigKey();
        orderTemplateConfigKey.setOrgCode(tempOrderCreateRequest.getOrganizationCode());
        orderTemplateConfigKey.setOrderMode(Contants.ORDER_MODE_NORMAL);
        // 获取订单处理模板工厂
        OrderTemplateFactory factory = orderTemplateFactoryUtil.getOrderTemplateFactory();

        List<TempOrderVo> tempOrderVos = new ArrayList<>();

        // 不同的GOODS_TYPE交由不同的处理器处理
        for (Map.Entry<String, List<GoodsInfoVo>> e : groupingGoods.entrySet()) {
            // 不同的GOODS_TYPE将对应不同的处理模板
            orderTemplateConfigKey.setGoodsType(e.getKey());
            // 通过工厂获取模板
            OrderTemplate orderTemplate = factory.getOrderTemplate(orderTemplateConfigKey);
            // 处理一下商品清单
            TempOrderCreateRequest requestNew = Beans.clone(tempOrderCreateRequest);
            requestNew.setGoodsInfoVos(e.getValue());
            requestNew.setGoodsType(e.getKey());
            // 走你！
            TempOrderVo tempOrderVo = orderTemplate.createTempOrder(requestNew);

            // 库存校验
            BaseResponse<OrderStock> stockRes = null;
            try {

                boolean stockFlag = true;
                for (TempSubOrderVo subOrder : tempOrderVo.getSubOrders()) {
                    stockRes = orderTemplate.validateStock(subOrder, tempOrderVo.getAreaCode(), tempOrderVo.getRealOrganizationId());
                    if (stockRes.isSuccess()) {
                        stockFlag = stockFlag && !stockRes.getResult().getHasNoStock();
                    } else {
                        throw new TempOrderCreateException(stockRes.getResultMessage() + "，错误代码" + stockRes.getResultCode(), Contants.TEMPORDER_CREATE_ERROR.AREA_CODE_ILLEGAL);
                    }
                }
                if (!stockFlag) {
                    throw new TempOrderCreateException(Contants.TEMPORDER_CREATE_ERROR.INCLUED_NO_STOCK);
                }
            } catch (IllegalAreaCodeException e1) {
                throw new TempOrderCreateException(e1.getMessage(), Contants.TEMPORDER_CREATE_ERROR.AREA_CODE_ILLEGAL);
            }

            // 根据费率重新计算销售价
            orgSalesPriceOperator.updateTempOrder(tempOrderVo);

            // 获取运费
            BaseResponse<OrderFreight> orderFreightResp = orderTemplate.getFreight(tempOrderVo);
            if (orderFreightResp.isSuccess()) {
                // 回填整单的运费
                tempOrderVo.setSumShippingFee(orderFreightResp.getResult().getBaseFreight());
                // 此处为降低代码复杂度，只按店铺回填运费
                if (!Lang.isEmpty(orderFreightResp.getResult().getSubOrderFreights())) {
                    orderFreightResp.getResult().getSubOrderFreights().forEach(it ->
                            tempOrderVo.getSubOrders().stream()
                                    // 根据订单号索引子单
                                    .filter(so -> so.getTmpOrderNo().equals(it.getOrderNo())).findFirst()
                                    // 回填运费，TempOrderVo和TempSubOrderVo的setSumShippingFee会同时计算needPayPrice
                                    .ifPresent(tempSubOrderVo -> tempSubOrderVo.setSumShippingFee(it.getBaseFreight())));
                }
            } else {
                // 抛出运费获取失败异常
                throw new TempOrderCreateException(orderFreightResp.getResultMessage(), Contants.TEMPORDER_CREATE_ERROR.FALID_ON_QUERYING_FERIGHT);
            }

            // 营销活动，营销活动会影响订单金额
            orderTemplate.marketingDone(tempOrderVo);
            tempOrderVo.setNeedPayPrice(tempOrderVo.getSumPrice().add(tempOrderVo.getSumShippingFee()));
            tempOrderVos.add(tempOrderVo);
        }

        // 整理一下createdTempOrder作为最终输出
        if (tempOrderVos != null && !tempOrderVos.isEmpty()) {
            if (tempOrderVos.size() > 1) {
                // 组合成多个一齐返回
                createdTempOrder = tempOrderHelper.combime(tempOrderVos);
            } else {
                createdTempOrder = tempOrderVos.get(0);
            }
        }

        return createdTempOrder;
    }

    /**
     * 临时订单库存校验
     *
     * @param tempOrderVo 临时订单对象
     */
    @Override
    public OrderStock tempOrderStock(TempOrderVo tempOrderVo) throws IllegalAreaCodeException {
        return orderStockOperator.get(tempOrderVo);
    }

    /**
     * 临时定案价格重算
     *
     * @param tempOrderVo
     * @return
     */
    @Override
    public OrderPrice tempOrderPrice(TempOrderVo tempOrderVo) throws IllegalAreaCodeException {
        orderFreightOperator.update(tempOrderVo);
        return orderPriceOperator.get(tempOrderVo);
    }

    @Override
    public OrderVo submit(TempOrderVo tempOrder) throws OrderCreateException {

        // 提交订单,flatSubOrders获取临时订单中所有的最末级子单，每个子单生成一个order_main
        List<TempSubOrderVo> tempSubOrders = TempOrderHelper.flatSubOrders(tempOrder);
        // 根据不同的子订单，分解生成orderMain
        OrderTemplateFactory factory = orderTemplateFactoryUtil.getOrderTemplateFactory();

        List<OrderVo> createdOrders = new ArrayList<>();

        for (TempSubOrderVo s : tempSubOrders) {
            OrderTemplateConfigKey configKey = new OrderTemplateConfigKey();
            configKey.setGoodsType(s.getGoodsType());
            configKey.setOrderMode(s.getOrderMode());
            configKey.setOrgCode(Lang.isEmpty(s.getOrgCode()) ? tempOrder.getOrganizationCode() : s.getOrgCode());

            OrderTemplate template = factory.getOrderTemplate(configKey);

            // 1. 调用orderPreCheck，订单的各种个性化校验
            BaseResponse<Boolean> preCheckRs = template.orderPreCheck(tempOrder);
            if (!preCheckRs.isSuccess() || !preCheckRs.getResult()) {
                throw new OrderCreateException(preCheckRs.getResultMessage(), Contants.ORDER_CREATE_ERROR.FAILD_ON_ORDER_PRECHECK);
            }

            try {
                // 订单金额校验及金额重算
                BigDecimal orginNeedPayPrice = tempOrder.getNeedPayPrice();
                // 2. 重新获取订单金额，以防信息被篡改
                //  2-1. 重新获取供应商售价
                OrderPrice orderPrice = orderPriceOperator.get(tempOrder);
                TempOrderHelper.updateTempOrderPrices(tempOrder, orderPrice);
                //  2-2. 重新计算费率浮动售价
//                orgSalesPriceOperator.updateTempOrder(tempOrder);
                //  2-3. 重新调用marketingDone，处理优惠、活动价格等营销活动处理
                BaseResponse<Boolean> marketingRs = template.marketingDone(tempOrder);
                if (!marketingRs.isSuccess()) {
                    throw new OrderCreateException(marketingRs.getResultMessage(), Contants.ORDER_CREATE_ERROR.FAILD_ON_MARKETING_PROCESSING);
                } else if (!marketingRs.getResult()) {
                    throw new OrderCreateException(Contants.ORDER_CREATE_ERROR.FAILD_ON_MARKETING_PROCESSING);
                }
                //  3. 重新获取运费
                BaseResponse<OrderFreight> orderFreightRs = template.getFreight(tempOrder);
                if (!orderFreightRs.isSuccess()) {
                    throw new OrderCreateException(orderFreightRs.getResultMessage(), Contants.ORDER_CREATE_ERROR.FAILD_ON_ORDER_FREIGHT);
                }
                tempOrder.setSumShippingFee(orderFreightRs.getResult().getBaseFreight());

                //如果刷新后的NeedPayPrice与刷新前不符，抛出异常，前端页面触发了价格刷新后，再提交
                if (orginNeedPayPrice.subtract(tempOrder.getNeedPayPrice()).abs().compareTo(BigDecimal.ZERO) > 0) {
                    OrderCreateException e = new OrderCreateException(Contants.ORDER_CREATE_ERROR.ORDER_PRICE_REFRESHED);
                    e.setData(tempOrder);
                    throw e;
                }
                CreateOrderRequest createOrderRequest = BeanMapper.map(tempOrder, BeanMapper.getType(TempOrderVo.class), BeanMapper.getType(CreateOrderRequest.class));
                createOrderRequest.setSubOrder(s);
                createOrderRequest.setPaymentCode(tempOrder.getPayMethodCode());
                OrderVo orderVo = null;
                try {
                    // 调用订单创建
                    orderVo = template.createOrder(createOrderRequest);
                    createdOrders.add(orderVo);
                    //TODO: 如非常强调异步性能，可改为异步方式，也可改为MQ方式实现，但需考虑MQ丢失消息的情况，可能会引起外围系统与订单的关联等数据被丢失
                    OrderMain order = orderMainQueryService.findByKey(new OrderKey(orderVo.getOrderId(), orderVo.getOrderNo()));
                    if (order != null) {
                        orderHelper.publishOrderEvent(new OrderCreateEvent(
                                order
                                , orderVo.getOrganizationCode()
                                , Contants.ORDER_OPERATOR_SYSTEM));
                    }

                } catch (NoStockException e) {
                    throw new OrderCreateException(Contants.ORDER_CREATE_ERROR.INCLUED_NO_STOCK);
                }
            } catch (IllegalAreaCodeException e) {
                throw new OrderCreateException(Contants.ORDER_CREATE_ERROR.AREA_CODE_ILLEGAL, e);
            }

        }

        List<OrderVo> orderVoList = createdOrders.stream().filter(it -> it != null).collect(Collectors.toList());

        //判断是否需要父订单
        OrderVo pOrder = null;
        if (orderVoList.size() > 1) {
            pOrder = BeanMapper.map(orderVoList.get(0), OrderVo.class);
            // 填充子单的item到父订单
            pOrder.setOrderItems(orderVoList.stream()
                    .flatMap(o -> o.getOrderItems().stream())
                    .collect(Collectors.toList()));
            pOrder.setSubOrders(orderVoList);
            pOrder.setOrderId(null);
            pOrder.setOrderNo(orderHelper.generateOrderNo());
            // 合并运费
            pOrder.setFreight(orderVoList.stream().map(OrderVo::getFreight).reduce(BigDecimal.ZERO, BigDecimal::add));
            // 合并无运费订单总价
            pOrder.setSumNofreightPrice(orderVoList.stream().map(OrderVo::getSumNofreightPrice).reduce(BigDecimal.ZERO, BigDecimal::add));
            pOrder.setSumPrice(pOrder.getSumNofreightPrice().add(pOrder.getFreight()));

        } else {
            pOrder = orderVoList.get(0);
            pOrder.setSubOrders(orderVoList);
        }
        pOrder.setTempOrderNo(tempOrder.getTmpOrderNo());

        return pOrder;
    }

    /**
     * 刷新订单状态
     *
     * @param orderKeys
     * @return
     */
    @Override
    public List<OrderVo> refresh(List<OrderKey> orderKeys) {
        assert (orderKeys != null && orderKeys.size() > 0) : "参数orderKeys不能为空";

        if (orderKeys == null || orderKeys.size() <= 0) {
            return null;
        }
        if (orderKeys.stream().filter(i -> Lang.isEmpty(i.getOrderNo()) || Lang.isEmpty(i.getOrderId())).count() <= 0) {
            return null;
        }

        orderKeys.stream().filter(i -> Lang.isEmpty(i.getOrderNo())).forEach(i -> {
            i.setOrderNo(orderRepos.findOrderNoById(i.getOrderId()));
        });

        // 获取订单处理模板工厂
        OrderTemplateFactory factory = orderTemplateFactoryUtil.getOrderTemplateFactory();

        List<OrderVo> r = new ArrayList<>();

        // 判断订单是否可进行审核
        List<OrderMain> orderMains = orderMainQueryService.findByKey(orderKeys);

        if (orderMains != null) {
            // 先将订单进行分组，适配批量处理
            Map<String, List<OrderMain>> groupedOrderVo = orderMains.stream()
                    .collect(Collectors.groupingBy(o ->
                            Joiner.on("_").join(new String[]{o.getOrderMode(), o.getGoodsType(), o.getRealOrganizationId()})));
            groupedOrderVo.entrySet().forEach(stringListEntry -> {
                List<OrderMain> orderVoList = stringListEntry.getValue();
                List<OrderKey> orderKeyList = orderVoList.stream().map(i -> new OrderKey(i.getId(), i.getOrderNo())).collect(Collectors.toList());
                if (orderVoList.size() > 0) {
                    OrderMain so = orderVoList.get(0);
                    OrderTemplateConfigKey configKey = new OrderTemplateConfigKey();
                    configKey.setOrderMode(Contants.ORDER_MODE_NORMAL);
                    configKey.setGoodsType(so.getGoodsType());
                    configKey.setOrgCode(organizationApi.findOrganizationCodeById(so.getRealOrganizationId()));

                    OrderTemplate template = factory.getOrderTemplate(configKey);

                    template.refresh(orderKeyList);
                }
            });
        }

        // 触发订单状态更改事件
//        orderMains.forEach(it -> {
//            orderHelper.publishOrderStatusChangeEvent(new OrderKey(it.getId(), it.getOrderNo()), null);
//        });
        for(OrderMain it : orderMains) {
            orderHelper.publishOrderStatusChangeEvent(new OrderKey(it.getId(), it.getOrderNo()), null);
        }

        return r;
    }

    @Override
    public OrderAuditingResult audit(OrderKey orderKey, Boolean passed, Boolean cancelSupplierOrder) {
        List<OrderAuditingResult> r = audit(new ArrayList<OrderKey>() {{
            add(orderKey);
        }}, passed, cancelSupplierOrder);
        return r == null && r.size() <= 0 ? null : r.get(0);
    }

    @Override
    public List<OrderAuditingResult> audit(List<OrderKey> orderKeys, Boolean passed, Boolean cancelSupplierOrder) {
        assert (orderKeys != null && orderKeys.size() > 0) : "参数orderKeys不能为空";
        assert (passed != null) : "参数passed不能为空";

        if (passed == null) {
            return null;
        }
        if (orderKeys == null || orderKeys.size() <= 0) {
            return null;
        }
        if (orderKeys.stream().filter(i -> !Lang.isEmpty(i.getOrderNo()) || !Lang.isEmpty(i.getOrderId())).count() <= 0) {
            return null;
        }

        orderKeys.stream().filter(i -> Lang.isEmpty(i.getOrderNo())).forEach(i -> {
            i.setOrderNo(orderMainQueryService.findOrderNoById(i.getOrderId()));
        });

        // 获取订单处理模板工厂
        OrderTemplateFactory factory = orderTemplateFactoryUtil.getOrderTemplateFactory();


        List<OrderAuditingResult> r = new ArrayList<>();

        // 判断订单是否可进行审核
        List<OrderMain> orderMains = orderMainQueryService.findByKey(orderKeys);

        if (orderMains != null) {
            // 先将订单进行分组，适配批量处理
            Map<String, List<OrderMain>> groupedOrderVo = orderMains.stream()
                    .collect(Collectors.groupingBy(o ->
                            Joiner.on("_").join(new String[]{o.getOrderMode(), o.getGoodsType(), o.getRealOrganizationId()})));
            groupedOrderVo.entrySet().forEach(stringListEntry -> {
                List<OrderMain> orderVoList = stringListEntry.getValue();
                if (orderVoList.size() > 0) {
                    OrderMain so = orderVoList.get(0);
                    OrderTemplateConfigKey configKey = new OrderTemplateConfigKey();
                    configKey.setOrderMode(Contants.ORDER_MODE_NORMAL);
                    configKey.setGoodsType(so.getGoodsType());
                    configKey.setOrgCode(organizationApi.findOrganizationCodeById(so.getRealOrganizationId()));

                    OrderTemplate template = factory.getOrderTemplate(configKey);
                    List<OrderKey> auditKeys = orderVoList.stream()
                            .map(orderMain -> new OrderKey(orderMain.getId(),orderMain.getOrderNo()))
                            .collect(Collectors.toList());
                    // 调用OrderTemplate进行处理
                    List<OrderAuditingResult> responseList = template.audit(auditKeys, passed);
                    List<OrderKey> sucList = responseList.stream()
                            .filter(OrderAuditingResult::getSuccess)
                            .map(OrderAuditingResult::getOrderKey).collect(Collectors.toList());

                    // 审核通过的，进行预占库存
                    if (sucList != null && sucList.size() > 0) {
                        sucList.forEach(sucKey -> {
                            Optional<OrderMain> optOrderMain = orderMains.stream()
                                    .filter(om -> om.getId().equals(sucKey.getOrderId())
                                            || om.getOrderNo().equals(sucKey.getOrderNo()))
                                    .findFirst();
                            if (optOrderMain.isPresent()) {
                                Payment payment = paymentService.findByid(optOrderMain.get().getPaymentId());
                                MemberAddressDto memberAddressDto = memberAddressApi.findById(optOrderMain.get().getAddressId());
                                BaseResponse<Boolean> subsucribeRs = template.subscribeOrder(orderMainService.toOrderVo(optOrderMain.get(), payment, memberAddressDto));
                                if (!subsucribeRs.isSuccess()) {
                                    // 预占库存失败
                                    responseList.stream().filter(r2 -> r2.getOrderKey().equals(sucKey)).findFirst().ifPresent(r2 -> {
                                        r2.setSuccess(false);
                                        r2.setMessage(subsucribeRs.getResultMessage());
                                        r2.setErrorCode(Contants.ORDER_AUDITING_ERROR.SUPPLIER_TACK_ORDER_ERROR.getValue());
                                    });
                                    List<OrderKey> cancelKeys = new ArrayList<>();
                                    cancelKeys.add(sucKey);
                                    template.cancelOrder(cancelKeys, Contants.CancelOrderType.ERROR);
                                }
                            } else {
                                // 出错了！
                                responseList.stream().filter(r2 -> r2.getOrderKey().equals(sucKey)).findFirst().ifPresent(r2 -> {
                                    r2.setSuccess(false);
                                    r2.setMessage(Contants.ORDER_AUDITING_ERROR.ORDERMAIN_NOT_FOUND.getMessage());
                                    r2.setErrorCode(Contants.ORDER_AUDITING_ERROR.ORDERMAIN_NOT_FOUND.getValue());
                                });
                            }


                        });

                    }
                    r.addAll(responseList);
                    if (!passed && cancelSupplierOrder) {
                        // 如果需要取消供应商订单的，在审核不通过后，调用供应商订单取消
                        template.cancelOrder(auditKeys, Contants.CancelOrderType.ERROR);
                    }
                }
            });
        }

        // 触发订单状态更改事件
//        orderMains.forEach(it -> {
//            orderHelper.publishOrderStatusChangeEvent(new OrderKey(it.getId(), it.getOrderNo()), null);
//        });

//        for(OrderMain it : orderMains) {
//            orderHelper.publishOrderStatusChangeEvent(new OrderKey(it.getId(), it.getOrderNo()), null);
//        }

        return r;
    }

    /**
     * 订单支付预处理完毕
     * （客户已开始支付，但未支付成功时调用）
     *
     * @param orderKey   订单key，orderNo或orderId，优先id
     * @param cusPayInfo 客户支付信息，如需在客户支付环节带入新的信息从此参数带入
     */
    @Override
    public void prePayDone(OrderKey orderKey, CusPayInfo cusPayInfo) {
        OrderMain orderMain = orderMainQueryService.findByKey(orderKey);
        // 获取订单处理模板工厂
        OrderTemplateFactory factory = orderTemplateFactoryUtil.getOrderTemplateFactory();
        OrderTemplateConfigKey configKey = new OrderTemplateConfigKey();
        configKey.setOrderMode(Lang.isEmpty(orderMain.getOrderMode()) ? Contants.ORDER_MODE_NORMAL : orderMain.getOrderMode());
        configKey.setGoodsType(orderMain.getGoodsType());
        configKey.setOrgCode(organizationApi.findOrganizationCodeById(orderMain.getRealOrganizationId()));

        Payment payment = paymentService.findByid(orderMain.getPaymentId());
        MemberAddressDto memberAddressDto = memberAddressApi.findById(orderMain.getAddressId());

        factory.getOrderTemplate(configKey).orderPayPreSubmitDone(orderMainService.toOrderVo(orderMain, payment, memberAddressDto));
    }

    @Override
    public OrderPayConfirmResp payComfirmed(OrderKey orderKey) {
        // 确认订单
        OrderMain orderMain = orderMainQueryService.findByKey(orderKey);
        // 获取订单处理模板工厂
        OrderTemplateFactory factory = orderTemplateFactoryUtil.getOrderTemplateFactory();
        OrderTemplateConfigKey configKey = new OrderTemplateConfigKey();
        configKey.setOrderMode(Lang.isEmpty(orderMain.getOrderMode()) ? Contants.ORDER_MODE_NORMAL : orderMain.getOrderMode());
        configKey.setGoodsType(orderMain.getGoodsType());
        configKey.setOrgCode(organizationApi.findOrganizationCodeById(orderMain.getRealOrganizationId()));

        Payment payment = paymentService.findByid(orderMain.getPaymentId());
        MemberAddressDto memberAddressDto = memberAddressApi.findById(orderMain.getAddressId());
        OrderPayConfirmResp r = null;

        try {
            OrderTemplate template = factory.getOrderTemplate(configKey);
            r = template.paySubmit(orderMainService.toOrderVo(orderMain, payment, memberAddressDto));
        } catch (SupplierInterfaceInvokeException e) {
            r.setSuccess(false);
            switch (e.getErrorCode()) {
                case NETWORK_CONNECTION_FAILD:
                default:
                    r.setErrorCode(Contants.ORDER_PAY_CONFIRM_ERROR.SUPPLIER_INTERFACE_ERROR);
            }

            e.printStackTrace();
        } catch (SupplierOrderComfirmFalidException e) {
            r.setSuccess(false);
            switch (e.getErrorCode()) {
                case ORDER_CANCELED:
                    r.setErrorCode(Contants.ORDER_PAY_CONFIRM_ERROR.ORDER_CANCELED);
                    break;
                case ACCOUNT_BALANCE_ERROR:
                    r.setErrorCode(Contants.ORDER_PAY_CONFIRM_ERROR.ACCOUNT_BALANCE_ERROR);
                    break;
                case ALREADY_COMFIRMED_ORDER:
                    r.setErrorCode(Contants.ORDER_PAY_CONFIRM_ERROR.ALREADY_COMFIRMED_ORDER);
                    break;
                default:
                    r.setErrorCode(Contants.ORDER_PAY_CONFIRM_ERROR.SUPPLIER_COMFIRMED_ERROR);
                    r.setMessage(e.getMessage());
            }


            e.printStackTrace();
        } catch (Exception e) {
            r.setSuccess(false);
            r.setErrorCode(Contants.ORDER_PAY_CONFIRM_ERROR.SUPPLIER_COMFIRMED_ERROR);
            r.setMessage(e.getMessage());
            e.printStackTrace();
        }


        return r;
    }

    /**
     * 订单取消
     *
     * @param orderKeys  其中orderNo、orderId可任选其一
     * @param cancelType
     */
    @Override
    public CancelOrderResp cancel(List<OrderKey> orderKeys, Contants.CancelOrderType cancelType) {
        AtomicLong faildCount = new AtomicLong();
        AtomicLong sucCount = new AtomicLong();
        List<CancelOrderDetailResp> cancelOrderDetailResps = new ArrayList<>();
        // 获取订单处理模板工厂
        OrderTemplateFactory factory = orderTemplateFactoryUtil.getOrderTemplateFactory();
        // 查找订单
        List<OrderMain> orderMains = orderMainQueryService.findByKey(orderKeys);
        if (orderMains != null) {
            // 先将订单进行分组，已适配批量取消场景
            Map<String, List<OrderMain>> groupedOrderVo = orderMains.stream()
                    .collect(Collectors.groupingBy(o ->
                            Joiner.on("_").join(new String[]{o.getOrderMode(), o.getGoodsType(), o.getRealOrganizationId()})));

            groupedOrderVo.entrySet().forEach(stringListEntry -> {
                List<OrderMain> orderVoList = stringListEntry.getValue();
                if (orderVoList.size() > 0) {
                    OrderMain so = orderVoList.get(0);
                    OrderTemplateConfigKey configKey = new OrderTemplateConfigKey();
                    configKey.setOrderMode(Contants.ORDER_MODE_NORMAL);
                    configKey.setGoodsType(so.getGoodsType());
                    configKey.setOrgCode(organizationApi.findOrganizationCodeById(so.getRealOrganizationId()));

                    OrderTemplate template = factory.getOrderTemplate(configKey);
                    List<OrderKey> cancelKeys = orderVoList.stream().map(orderMain -> {
                        OrderKey key = new OrderKey();
                        key.setOrderId(orderMain.getId());
                        return key;
                    }).collect(Collectors.toList());
                    BaseResponse<CancelOrderResp> response = template.cancelOrder(cancelKeys, cancelType);
                    cancelOrderDetailResps.addAll(response.getResult().getDetails());
                    faildCount.addAndGet(response.getResult().getFailureCounts());
                    sucCount.addAndGet(response.getResult().getSuccessCounts());
                }
            });
        } else {
            //TODO: 错误处理

        }
        CancelOrderResp cancelOrderResp = new CancelOrderResp();
        cancelOrderResp.setDetails(cancelOrderDetailResps);
        cancelOrderResp.setSuccessCounts(sucCount.get());
        cancelOrderResp.setFailureCounts(faildCount.get());

        // 处理订单事件（只处理成功的订单）
        List<String> canceledOrderNos = cancelOrderDetailResps.stream()
                .filter(i -> i.getSuccessed())
                .map(CancelOrderDetailResp::getOrderNo)
                .collect(Collectors.toList());
        List<OrderMain> canceledOrderMain = orderMainQueryService.findByOrderNos(canceledOrderNos);

        for (String orderNo : canceledOrderNos) {
            canceledOrderMain.stream()
                    .filter(i -> orderNo.equals(i.getOrderNo()))
                    .findFirst()
                    .ifPresent(orderMain -> orderHelper.publishOrderEvent(new OrderStatusChangeEvent(
                            orderMain
                            , organizationApi.findOrganizationCodeById(orderMain.getOrganizationId())
                            // 区分是否客户主动取消
                            , cancelType.equals(Contants.CancelOrderType.USER) ? orderMain.getMemberId() : Contants.ORDER_OPERATOR_SYSTEM
                    )));
        }

        return cancelOrderResp;
    }

    /**
     * 订单收货
     *
     * @param orderKey    其中orderNo、orderId可任选其一
     * @param receiptType 订单收货类型，见{@link Contants}中ORDER_RECEIPT_TYPE_开头
     * @return
     */
    @Override
    public void receipt(OrderKey orderKey, String receiptType) {
        OrderMain orderMain = orderMainQueryService.findByKey(orderKey);
        String oldStatus = orderMain.getStatus();
        // 获取订单处理模板工厂
        OrderTemplateFactory factory = orderTemplateFactoryUtil.getOrderTemplateFactory();
        OrderTemplateConfigKey configKey = new OrderTemplateConfigKey();
        configKey.setOrderMode(Lang.isEmpty(orderMain.getOrderMode()) ? Contants.ORDER_MODE_NORMAL : orderMain.getOrderMode());
        configKey.setGoodsType(orderMain.getGoodsType());
        configKey.setOrgCode(organizationApi.findOrganizationCodeById(orderMain.getRealOrganizationId()));
        OrderTemplate template = factory.getOrderTemplate(configKey);
        if (template != null) {
            template.orderReceipt(orderKey, receiptType);
        }
        orderMain = orderMainQueryService.findByKey(orderKey);
        // 触发订单确认收货事件
        if (!oldStatus.equals(orderMain.getStatus())
                && (orderMain.getStatus().equals(GlobalContants.ORDER_STATUS.EVALUATE.getValue())
                || orderMain.getStatus().equals(GlobalContants.ORDER_STATUS.FINISH.getValue()))) {
            String orgCode = organizationApi.findOrganizationCodeById(orderMain.getOrganizationId());
            orderHelper.publishOrderEvent(new OrderShippingComfirmedEvent(orderMain, orgCode, Contants.ORDER_OPERATOR_SYSTEM));
        }
    }

    @Override
    public void statusChange(OrderStatus orderStatus) {
        OrderMain orderMain = orderMainQueryService.findByKey(new OrderKey(orderStatus.getOrderId(), orderStatus.getOrderNo()));
        if(orderMain != null) {
            String oldStatus = orderMain.getStatus();
            if (!Lang.isEmpty(orderStatus.getOrderId())) {
                orderRepos.updateStatusById(orderStatus.getStatus().getValue(), orderStatus.getOrderId());
            } else if (!Lang.isEmpty(orderStatus.getOrderNo())) {
                orderRepos.updateStatusByOrderNo(orderStatus.getStatus().getValue(), orderStatus.getOrderNo());
            }
            if (!oldStatus.equals(orderStatus.getStatus().getValue())) {
                String orgCode = organizationApi.findOrganizationCodeById(orderMain.getOrganizationId());
                orderHelper.publishOrderEvent(new OrderStatusChangeEvent(orderMain, orgCode, Contants.ORDER_OPERATOR_SYSTEM));
            }
        }
    }
}
