package order.service.stubs;

import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import outsideapi.api.OutsideProductApi;
import outsideapi.vo.*;
import store.api.dto.modeldto.core.StoreDto;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

/**
 * 供应商产品接口本地代理
 * @author Liang Wenxu
 * @since 2018/7/27
 */
@Slf4j
@Component
@Transactional(propagation = Propagation.NOT_SUPPORTED)
public class OutsideProductApiStub {
    /**
     * 获取第三方店铺商品实时库存
     * @param store
     * @param requetVo
     * @return HandlerRespVo<List < FashionStockStateVo>>
     *     错误代码errorCode定义： 1-- 请求参数错误 2-- 商户接口Handler未配置
     *     FashionStockStateVo 中的主要字段使用提醒：
     *          onSale： 标识是否可售状态，可能各个供应商其具体含义不同，当此值为false时商品应显示为不可售状态
     *          stockCount： 库存，当返回-1时为不限制库存（需结合onSale进行判断）
     *          isAreaRestrict： 当前地区是否限制购买，京东特有，其它供应商都将统一返回false，不对地区进行控制
     *          stockFlag：是否有货 0-- 无货，1--有货，2--预定（延后发货）
     */
    public HandlerRespVo<List<FashionStockStateVo>> getFashionStockState(StoreDto store, FashionStatusRequetVo requetVo) {
        return outsideProductApi.getFashionStockState(store, requetVo);
    }

    /**
     * 判断是否外部第三方店铺（实体类）
     * @param store 店铺信息实体类
     * @return HandlerRespVo.data返回true时为第三方商铺
     */
    public HandlerRespVo<Boolean> isOutsideStore(StoreDto store) {
        return outsideProductApi.isOutsideStore(store);
    }

    /**
     * 判断是否外部第三方店铺
     * @param storeId Store店铺信息表ID
     * @return HandlerRespVo.data返回true时为第三方商铺
     */
    public HandlerRespVo<Boolean> isOutsideStore(String storeId) {
        return outsideProductApi.isOutsideStore(storeId);
    }

    /**
     * 获取商品用于费率查询的区间基准值，京东为 协议价 / 市场价
     * @param requestVo 请求vo
     * @return 按requestVoList的索引顺序返回费率区间基准值
     *      <br>错误代码errorCode定义：
     *      <br>&nbsp;&nbsp;&nbsp;&nbsp;2 -- 商户接口Handler未配置
     *      <br>&nbsp;&nbsp;&nbsp;&nbsp;其它值 -- 其它错误
     *      <br>返回数据data: 用于费率查询的区间基准值
     */
    public HandlerRespVo<BigDecimal> getSalesPriceRateRangeVal(FashionPricesRequestVo requestVo) {
        return outsideProductApi.getSalesPriceRateRangeVal(requestVo);
    }

    /**
     * 批量获取商品用于费率查询的区间基准值，京东为 协议价 / 市场价
     * @param requestVoList 请求VO列表（批量查询）
     * @return 按requestVoList的索引顺序返回费率区间基准值
     *      <br>错误代码errorCode定义：
     *      <br>&nbsp;&nbsp;&nbsp;&nbsp;2 -- 商户接口Handler未配置
     *      <br>&nbsp;&nbsp;&nbsp;&nbsp;其它值 -- 其它错误
     *      <br>返回数据data: 用于费率查询的区间基准值
     */
    public HandlerRespVo<List<BigDecimal>> getBatchSalesPriceRateRangeVal(List<FashionPricesRequestVo> requestVoList) {
        return outsideProductApi.getBatchSalesPriceRateRangeVal(requestVoList);
    }

    /**
     * 查询第三方商户配送费用
     * @param storeId
     * @param requet
     * @return 成功时返回data为运费信息，失败时返回错误代码errorCode及错误信息message
     *      <br>错误代码errorCode定义：
     *      <pre>&#9;2 -- 商户接口Handler未配置
     *      <br>&#9;其它代码 -- 其它错误
     *      </pre>
     *      <br>返回的data示意：
     *      <pre>data:{
     *      <br>&#9;freight: BigDecimal, // 总运费
     *      <br>&#9;baseFreight: BigDecimal, // 基础运费
     *      <br>&#9;remoteRegionFreight: BigDecimal, // 偏远运费
     *      <br>&#9;remoteFashions: String, // 需收取偏远运费的Fashion的id,多个ID使用","连接
     *      <br>}</pre>
     */
    public HandlerRespVo<Map<String, Object>> queryShippingFee(String storeId, ShippingFeeRequetVo requet) {
        return outsideProductApi.queryShippingFee(storeId, requet);
    }

    /**
     * 查询第三方实时价格接口
     * @param storeId 第三方商户Store的ID
     * @param request 请求参数Vo
     * @return 成功时返回data为运费信息，失败时返回错误代码errorCode及错误信息message
     *      <br>错误代码errorCode定义：
     *      <pre>&#9;2 -- 商户接口Handler未配置
     *      <br>&#9;其它代码 -- 其它错误
     *      </pre>
     *      <br>返回的data为FashionPriceVo的集合
     */
    public HandlerRespVo<List<FashionPriceVo>> queryFashionPrice(String storeId, FashionPriceRequestVo request) {
        return outsideProductApi.queryFashionPrice(storeId, request);
    }

    /**
     * 查询第三方物流
     * @param orderNo 集采订单号
     * @return
     */
    public HandlerRespVo<List<OrderShippingPackageVo>> getShipingTrace(String orderNo) {
        return outsideProductApi.getShipingTrace(orderNo);
    }

    /**
     * 查询供应商订单状态
     * 返回的data为供应商订单状态状态值定义：
     *
     * @param orderNo
     */
    public HandlerRespVo<String> orderStatus(String orderNo) {
        return outsideProductApi.orderStatus(orderNo);
    }

    /**
     * 将集采通用地区编码集合转换为第三方地区编码集合
     *
     * @param storeId
     * @param areaCodeMap 集采通用地区编码集合，key为层级，同AddressCommonApi.AddressLevel各层级的值，value为地区代码
     * @return 第三方接口约定地区编码集合，key为层级，同AddressCommonApi.AddressLevel各层级的值，value为地区代码
     */
    public HandlerRespVo<Map<Integer, String>> transAreaCodeMap(String storeId, Map<Integer, String> areaCodeMap) {
        return outsideProductApi.transAreaCodeMap(storeId, areaCodeMap);
    }

    /**
     * 获取多个订单的物流可以是不同的供应商订单
     * @param orderNos
     * @return
     */
    public HandlerRespVo<List<OrderShippingPackageVo>> getManyOrderShipingTrace(List<String> orderNos) {
        return outsideProductApi.getManyOrderShipingTrace(orderNos);
    }

    /**
     * 将集采通用地区编码集合转换为第三方地区编码集合
     *
     * @param storeKey
     * @param requestVos 集采通用地区编码集合，key为层级，同AddressCommonApi.AddressLevel各层级的值，value为地区代码
     * @return 第三方接口约定地区编码集合，key为层级，同AddressCommonApi.AddressLevel各层级的值，value为地区代码
     */
    public HandlerRespVo<List<BatchAreaRequestVo>> batchTransAreaCodeMap(String storeKey, List<BatchAreaRequestVo> requestVos) {
        return outsideProductApi.batchTransAreaCodeMap(storeKey, requestVos);
    }

    @MotanReferer
    OutsideProductApi outsideProductApi;


}
