package order.service.utils;

import order.service.factory.BatchOrderTemplateFactory;
import order.service.factory.OrderTemplateFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;

/**
 * 订单处理模板工厂工具类
 * @author akers
 */
@Component
public class OrderTemplateFactoryUtil implements ApplicationContextAware {
    private static final String ORDER_FACTORY_BEAN_CONFIG_KEY = "orderservice.ordertemplate.factor.beanname";
    private static final String BATCH_ORDER_FACTORY_BEAN_CONFIG_KEY = "orderservice.batchordertemplate.factor.beanname";

    private ApplicationContext applicationContext;

    @Autowired
    private Environment env;

    /**
     * 根据配置获取工厂实例
     * @return
     */
    public OrderTemplateFactory getOrderTemplateFactory() {
        OrderTemplateFactory f = getFactory(readProp(ORDER_FACTORY_BEAN_CONFIG_KEY), OrderTemplateFactory.class);
        if(f == null){
            throw new RuntimeException("the OrderTemplateFactory named " + readProp(ORDER_FACTORY_BEAN_CONFIG_KEY) + " is NOT FOUND!");
        }
        return f;
    }

    /**
     * 根据配置获取工厂实例
     * @return
     */
    public BatchOrderTemplateFactory getBatchOrderTemplateFactory() {
        BatchOrderTemplateFactory f = getFactory(readProp(BATCH_ORDER_FACTORY_BEAN_CONFIG_KEY), BatchOrderTemplateFactory.class);
        if(f == null){
            throw new RuntimeException("the BatchOrderTemplateFactory named " + readProp(BATCH_ORDER_FACTORY_BEAN_CONFIG_KEY) + " is NOT FOUND!");
        }
        return f;
    }

    /**
     * 根据BeanName获取工厂实例
     * @param factoryBeanName 工厂实现类的BeanName
     * @return 返回工厂实例，其类型通过factoryClass决定
     */
    public <T> T getFactory(String factoryBeanName, Class<T> factoryClass) {
        T factory = factoryClass.cast(applicationContext.getBean(factoryBeanName));
        return factory;
    }

    public String readProp(String key) {
        return env.getProperty(key);
    }


    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
