package provider.api.gateway.filter;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.oauth2.provider.ClientDetails;
import org.springframework.security.oauth2.provider.ClientDetailsService;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.concurrent.TimeUnit;

/**
 * @author torvalds on 2018/7/5 18:07.
 * @version 1.0
 */
public class ProviderInfoWapperFilter implements Filter {
    private static final Logger logger = LoggerFactory.getLogger(ProviderInfoWapperFilter.class);
    RedisTemplate redisTemplate;
    ClientDetailsService clientDetailsService;
    private final static String OAUTH_INFO_PROVIDER_ID_REDIS_KEY = "MALL-PROVIDER-API-GATEWAY:OAUTH_INFO_PROVIDER_ID_REDIS_KEY";

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {

    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication == null) {
            logger.error("认证信息为空");
            throw new AccessDeniedException("认证信息为空");
        }
        User principal = (User) authentication.getPrincipal();
        BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(OAUTH_INFO_PROVIDER_ID_REDIS_KEY + ":" + principal.getUsername());
        String providerId = null;
        if (boundValueOperations.get() != null) {
            providerId = String.valueOf(boundValueOperations.get());
        } else {
            ClientDetails clientDetails = clientDetailsService.loadClientByClientId(principal.getUsername());
            Object providerIdObj = clientDetails.getAdditionalInformation().get("providerId");
            if (providerIdObj != null) {
                providerId = String.valueOf(providerIdObj);
            }
            boundValueOperations.set(providerId, 1, TimeUnit.DAYS);
        }

        ProviderRequestInfoWrapper providerRequestInfoWrapper = new ProviderRequestInfoWrapper((HttpServletRequest) servletRequest);
        providerRequestInfoWrapper.setAppId(principal.getUsername());
        if (StringUtils.isNoneEmpty(providerId)) {
            providerRequestInfoWrapper.setProviderId(providerId);
        }
        filterChain.doFilter(providerRequestInfoWrapper, servletResponse);
    }

    @Override
    public void destroy() {

    }

    public RedisTemplate getRedisTemplate() {
        return redisTemplate;
    }

    public void setRedisTemplate(RedisTemplate redisTemplate) {
        this.redisTemplate = redisTemplate;
    }


    public ClientDetailsService getClientDetailsService() {
        return clientDetailsService;
    }

    public void setClientDetailsService(ClientDetailsService clientDetailsService) {
        this.clientDetailsService = clientDetailsService;
    }
}
